// Description: detects a wand pointing at any geometry below it and produces an event
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> all wand events are also generated by this node (ex. button1)
//</ul>
//
// Category: Trigger
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfGroup.h>
#include <set>
#include "ygWorld.h"
#include "ygUser.h"
#include "ygWand.h"
#include "pointAtTrigger.h"

using namespace std;

extern "C" ygNode* construct_pointAtTrigger(const char* name,bool master) { return new pointAtTrigger(name,master); }

struct _pointAtTriggerPrivateData
	{
	set<ygWand*> prevpointing;
	float distance;
	};


pointAtTrigger::pointAtTrigger(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("pointAtTrigger");
	p_ = new struct _pointAtTriggerPrivateData;
	p_->distance = 10;
	}

pointAtTrigger::~pointAtTrigger(void)
	{
	delete p_;
	}


void pointAtTrigger::reset(void)
	{
	p_->prevpointing.clear();
	//set distance to 10.0
	p_->distance = 10;
	ygNode::reset();
	}


void pointAtTrigger::message(const ygMessage& msg)
	{
	//set the intersection distance from the wand
	if (msg == "distance")
		p_->distance = msg.floatArg(0);
	else
		ygNode::message(msg);
	}


void pointAtTrigger::app(void)
	{
	set<ygWand*> pointing;
	int i,j,f;
	pfMatrix myXform, myXformInv;
	getTransform(myXform);
	myXformInv.invertAff(myXform);
	//for all users in the world
	for (i=0; i < ygWorld::World->numUsers(); i++)
		{
		//for each user wand
		ygUser * user = ygWorld::World->user(i);
		for (j=0; j < user->numWands(); j++)
			{
			if ((user->wand(j)) && (user->wand(j)->isOfClass("ygWand")))
				{
				ygWand * wand = (ygWand*) user->wand(j);
				//if the wand is pointing at geometry
				if ((wand) && (pointingAtMe(wand,myXformInv)))
					{
					pointing.insert(pointing.begin(),wand);
					//if not previously pointing generate start event
					if (p_->prevpointing.find(wand) == p_->prevpointing.end())
						{
						ygString args("wand=");
						args += wand->name();
						args += " user=";
						args += user->name();
						//a wand and associated user have begun pointing
						eventOccurred("start",args);
						}
					//if the number of wand flags is positive
					if (wand->numFlags() > 0)
						{
						ygString event;
						ygString args("wand=");
						args += wand->name();
						args += " user=";
						args += user->name();
						//for each wand flag
						for (f=0; f < wand->numFlags(); f++)
							{
							wand->getFlag(f,event);
							//any wand events are generated while in the space
							eventOccurred(event,args);
							}
						}
					}
				}
			}
		}
	set<ygWand*>::const_iterator iter;
	//for each wand previously pointing at geometry
	for (iter = p_->prevpointing.begin(); iter != p_->prevpointing.end();
	     ++iter)
		//if not currently pointing then generate stop event
		if (pointing.find(*iter) == pointing.end())
			{
			ygString args("wand=");
			args += (*iter)->name();
			args += " user=";
			args += (*iter)->user()->name();
			//a wand and associated user have stopped pointing
			eventOccurred("stop",args);
			}
	p_->prevpointing = pointing;
	ygNode::app();
	}


bool pointAtTrigger::pointingAtMe(const ygWand* wand,const pfMatrix& myXformInv) const
	{
	pfSegSet segset;
	pfHit **hits[32];
	pfMatrix wandXform;
	pfVec3 forward(0, 1, 0), wandDir;
	segset.activeMask = 1;
	segset.isectMask = 0xFFFF;
	segset.discFunc = NULL;
	segset.bound = NULL;
	segset.mode = PFTRAV_IS_PRIM;
	segset.segs[0].pos = wand->origin(this);
	wand->getTransform(wandXform);
	wandDir.xformVec(forward, wandXform);
	wandDir.xformVec(wandDir, myXformInv);
	segset.segs[0].dir = wandDir;
	segset.segs[0].length = p_->distance;
	return pfnode()->isect(&segset, hits);
	}
