// Description: detects a wand pointing at any geometry below it and produces position events
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> all wand events are also generated by this node (ex. button1)
//<li> the local mode determines if the intersection position is local to the object or relative to world coordinates
//</ul>
//
// Category: Trigger
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pr/pfGeoSet.h>
#include <ygWorld.h>
#include <ygUser.h>
#include <ygWand.h>
#include "pointAtPosition.h"

extern "C" ygNode* construct_pointAtPosition(const char* name,bool master) { return new pointAtPosition(name,master); }

pointAtPosition::pointAtPosition(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("pointAtPosition");
	debugFlag("pointing", &debugPointing);
	distance = 10;
	}

pointAtPosition::~pointAtPosition(void)
	{
	}


void pointAtPosition::reset(void)
	{
	prevpointing.clear();
	//set reporting mode to local
	local = true;
	//set distance to 10.0
	distance = 10;
	ygNode::reset();
	}


void pointAtPosition::message(const ygMessage& msg)
	{
	//set the intersection distance from the wand
	if (msg == "distance")
		{
		distance = msg.floatArg(0);
		}
	//set position reporting mode
	else if (msg == "local")
		{
		if (msg.args.size() > 0)
			local = msg.boolArg(0);
		else
			local = true;
		}
	else
		ygNode::message(msg);
	}


void pointAtPosition::app(void)
	{
	set<ygWand*> pointing;
	int i,j,f;
	pfHit **hits[32];
	pfMatrix myXform, myXformInv;
	getTransform(myXform);
	myXformInv.invertAff(myXform);
	//for all users in the world
	for (i=0; i < ygWorld::World->numUsers(); i++)
		{
		//for each user wand
		ygUser * user = ygWorld::World->user(i);
		for (j=0; j < user->numWands(); j++)
			{
			if (user->wand(j) && user->wand(j)->isOfClass("ygWand"))
				{
				ygWand * wand = (ygWand*) (user->wand(j));
				//if the wand is pointing at geometry
				if ((wand) && (pointingAtMe(wand,myXformInv,hits)))
					{
					pointing.insert(pointing.begin(),wand);
					pfVec3 pnt;
					pfMatrix xmat;
					(*hits[0])->query(PFQHIT_POINT, pnt.vec);
					if (!local)
						{
						(*hits[0])->query(PFQHIT_XFORM, (float*)xmat.mat);
						pnt.xformPt(pnt, xmat);
						}
					if (debugPointing)
						cout << name() << ": " << user->name() << " is pointing\n";
					ygString args("wand=");
					args += wand->name();
					args += " user=";
					args += user->name();
					char value[32];
					args += " xpos=";
					sprintf(value,"%f",pnt[0]);
					args += value;
					args += " ypos=";
					sprintf(value,"%f",pnt[1]);
					args += value;
					args += " zpos=";
					sprintf(value,"%f",pnt[2]);
					args += value;
					//a user is pointing at the geometry
					eventOccurred("pointing",args);
					//if the number of wand flags is positive
					if (wand->numFlags() > 0)
						{
						ygString event;
						//for each wand flag
						for (f=0; f < wand->numFlags(); f++)
							{
							wand->getFlag(f,event);
							//any wand events are generated while in the space
							eventOccurred(event,args);
							}
						}
					//if not previously pointing generate start event
					if (prevpointing.find(wand) == prevpointing.end())
						{
						//a wand and associated user have begun pointing
						eventOccurred("start",args);
						}
					}
				}
			}
		}
	set<ygWand*>::const_iterator iter;
	//for each wand previously pointing at geometry
	for (iter = prevpointing.begin(); iter != prevpointing.end();
	     ++iter)
		//if not currently pointing then generate stop event
		if (pointing.find(*iter) == pointing.end())
			{
			ygString args("wand=");
			args += (*iter)->name();
			args += " user=";
			args += (*iter)->user()->name();
			//a wand and associated user have stopped pointing
			eventOccurred("stop",args);
			}
	prevpointing = pointing;
	ygNode::app();
	}


bool pointAtPosition::pointingAtMe(const ygWand* wand,const pfMatrix& myXformInv,pfHit*** hits) const
	{
	pfSegSet segset;
	pfMatrix wandXform;
	pfVec3 forward(0, 1, 0), wandDir;
	segset.activeMask = 1;
	segset.isectMask = 0xFFFF;
	segset.discFunc = NULL;
	segset.bound = NULL;
	segset.mode = PFTRAV_IS_PRIM;
	segset.segs[0].pos = wand->origin(this);
	wand->getTransform(wandXform);
	wandDir.xformVec(forward, wandXform);
	wandDir.xformVec(wandDir, myXformInv);
	segset.segs[0].dir = wandDir;
	segset.segs[0].length = distance;
	return pfnode()->isect(&segset, hits);
	}
