// Description: a transform that orients to face a node about a point or axis
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the model should be oriented with the negative Y axis
//<li> the point mode rotates the model about a point
//<li> the optional axis is the model up vector in point mode
//<li> the axis mode rotates the model about an axis
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/01
//
#include <Performer/pf/pfBillboard.h>
#include <ygUser.h>
#include <ygWorld.h>
#include <ygNodeDB.h>
#include "pointAtNode.h"

extern "C" ygNode* construct_pointAtNode(const char* name,bool master) { return new pointAtNode(name,master); }

pointAtNode::pointAtNode(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	setClassName("pointAtNode");
	axis.set(0,0,1);
	mode = PFBB_POINT_ROT_EYE;
	node = NULL;
	}

void pointAtNode::reset(void)
	{
	//set the rotation mode to point
	mode = PFBB_POINT_ROT_EYE;
	//set the rotation axis to the Z-axis
	axis.set(0,0,1);
	//clear node name
	node = NULL;
	ygSimpleTransform::reset();
	}


void pointAtNode::message(const ygMessage& msg)
	{
	//clear or set the node and optional user body part
	if (msg == "node")
		{
		//if number of arguments is greater than zero
		if (msg.args.size() > 0)
			{
			node = ygNodeDB::find(msg.args[0]);
			//if node type is user then
			if (msg.args.size() > 1 && node && node->isOfClass("ygUser"))
				{
				ygUser* user = (ygUser*)node;
				//if second argument is "head"
				if (msg.args[1] == "head")
					{
					//set node to user head
					if (user->head())
						node = user->head();
					}
				//if second argument is "body"
				else if (msg.args[1] == "body")
					{
					//for each user child
					for (int i=0; i<user->numChildren(); i++)
						{
						ygNode* currNode = user->child(i);
						//if child is of type body then set node
						if (currNode->isOfClass("body"))
							node = currNode;
						}
					}
				//if second argument is "wand"
				else if (msg.args[1] == "wand")
					{
					//if user has a wand then
					if (user->wand(0))
						{
						//use specified wand number or default to zero
						if (msg.args.size() > 2)
							node = user->wand(msg.intArg(1));
						else
							node = user->wand(0);
						}
					}
				else
					node = user;
				}
			}
		//else, clear the node name
		else
			node = NULL;
		}
	//set the mode and optional axis
	else if (msg == "mode")
		{
		if (msg.args.size() > 0)
			{
			mode = PFBB_AXIAL_ROT;
			if (msg.args[0] == "point")
				mode = PFBB_POINT_ROT_EYE;
			else if (msg.args[0] == "axis")
				mode = PFBB_AXIAL_ROT;
			if (msg.args.size() > 1)
				{
				msg.getVec3Args(axis,1);
				axis.normalize();
				}
			else
				axis.set(0,0,1);
			}
		else
			{
			mode = PFBB_AXIAL_ROT;
			axis.set(0,0,1);
			}
		}
	else
		ygSimpleTransform::message(msg);
	}


void pointAtNode::app(void)
	{
	//if a node is set
	if (node) 
		{
		//get node position vector relative to this node
		pfVec3 nodeVector = node->origin(parent()); 
		nodeVector.normalize();

		//get the cross product of the node vector and the rotation axis
		pfVec3 nodeCross;
		nodeCross.cross(axis,nodeVector);
		nodeCross.normalize();

		//establish the object front as the negative Y-axis
		pfVec3 frontAxis(0,1,0);
		pfVec3 frontCross;
		//get the cross product of the front axis and the rotation axis
		frontCross.cross(axis,frontAxis);
		frontCross.normalize();
		float adjacent = frontCross.dot(nodeCross);
		pfVec3 thirdNormal;
		//get a normal towards the nodeer by crossing this cross with the rotaion axis
		thirdNormal.cross(axis,frontCross);
		float opposite = thirdNormal.dot(nodeCross);
		//get the rotation to the node vector
		float axisRot = pfArcTan2(opposite,adjacent);
		//create a matrix with this rotation
		pfMatrix matrix;
		matrix.makeRot(axisRot,axis[0],axis[1],axis[2]);

		//if the mode if point then
		if (mode == PFBB_POINT_ROT_EYE)
			{
			float crossRot;
			//get the rotation about the nodeer cross product
			crossRot = pfArcSin(axis.dot(nodeVector));
			//set the final matrix to this rotation
			pfMatrix pitchMat;
			pitchMat.makeRot(-crossRot,nodeCross[0],nodeCross[1],nodeCross[2]);
			//add this rotation matrix to the existing rotation matrix
			matrix.postMult(pitchMat);
			}

		//set the dcs matrix
		setMatrix(matrix);
		}
	ygSimpleTransform::app();
	}
