// Description: a transform that will move an object throught a random path inside a volume
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the volume message sets the volume in which object(s) move with a random path
//<li> box( minX minY minZ maxX maxY maxZ)
//<ol>
//<li> set absolute box coordinates (volume) in which object(s) move with a random path
//<li> For the time being the inital position of the object will be minX+(maxX-minX)/2, etc. (box center)
//<li> Default 'box' is half CAVE size (see contructor).
//<li> WARNING: When user specify a new box (volume) the initial position is automatically re-set to a new point in the center of the new box. All random points following this initial position are also recalculated
//</ol>
//<li> randompoints 
//<ol>
//<li> sets the number of random points generated in the box and slso affects the speed
//<li> (as opposite you might think, the less random points the slower the movement)
//</ol>
//<li>allrot(min_roll min_pitch min_heading max_roll max_pitch max_heading rot_speed_roll rot_speed_pitch rot_speed_heading)
//<ol>
//<li> Normaly the vector-direction of the object (orientation) following the random path is the tangent of that path
//<li> If this message and parameters are set, the vector-direction will rotate between min max range, like swinging between those ranges
//<li> To maintain 1/1000 degree resolution in the single precision arithmetic used internally, for sine and cosine calculations, the angles head, pitch, roll should be in the range of -7500 to +7500 degrees.
//<li><b>Ex:</b> allrot -> enable rotation with default range and speed allrot(-10 -90 -80 +180 +180 +180  0.02 0.01 0.015) -> enable rot and set range & speed
//</ol>
//<li> every time a new postion is set, the app sends an event with the new coordinates 
//<li> events can be used by other nodes to set/rest their position
//</ul>
//
// Category: Transformation
// Author: Javier I. Girado, Brenda Lopez
// Revision: 11/10/01 
//
#include <vector>
#include <ygUtil.h>
#include <ygWorld.h>
#include "pathRandom.h"
//#define DEBUG
using namespace std;

extern "C" ygNode* construct_pathRandom( const char* name, bool master) 
{ 
	return( new pathRandom(name,master) ); 
}

struct _pathRandomPrivateData
{
	bool active;

    vector<pfVec3*> points;
	
	unsigned int random_points;
	
	float init_x, init_y, init_z;

	float minX, maxX,  
		  minY, maxY, 
		  minZ, maxZ;

	float index, speed;		  
		  
	bool  allrot;
	
	float roll_min, roll_max,  
		  pitch_min,pitch_max, 
		  head_min, head_max;

	float pitch_speed, roll_speed, head_speed;
	float pitch_index, roll_index, head_index;

	pfVec3 pos, dir;
};


class pathRandomCatmullRomCurve
{
public:
	 pathRandomCatmullRomCurve( void );
	 
	 void   setControlPoints( pfVec3, pfVec3, pfVec3, pfVec3 );
	 pfVec3 value(   float t );
	 pfVec3 tangent( float t );
	 
protected:

	 void   computeCoefficients( void );
	 pfVec3 cpt_[4];
	 pfVec4 coeff_[3];
};

//
// Static functions
//

static float randomVal(float min, float max) { return (min + drand48()*(max - min)); };

//
// standard Yg template functions
//
pathRandom::pathRandom( const char* name, bool master) : ygTransform( name, master)
{
	setClassName( "pathRandom" );
	
	p_ = new struct _pathRandomPrivateData;
	
	reset();
}

pathRandom::~pathRandom(void)
{
	delete p_;
	p_->active = false;
}

// This is called the begining (and when user press reset)
//
void pathRandom::reset(void)
{
	ygTransform::reset();

	p_->active = false;
	
	///////////////////////////
	// Defaults values       //
	///////////////////////////
	
	// 
	p_->minX = -5.0; p_->maxX = +5.0;
	p_->minY = +2.5; p_->maxY = +5.0;
	p_->minZ = +5.0; p_->maxZ = 10.0;
	
	// set number of random points at 100
	p_->random_points = 100;
	
	p_->index = 0.0;
	// set speed to 0.001
	p_->speed = 0.001;

	setInitPos( ); // First, determine initial position based on box dimensions

	loadPath( p_->random_points ); // Add to that initial position a series of

	// Defualt random rotation						   
	p_->allrot = false;
	
	// set roll_min to -180, pitch_min to -180, head_min to -180
	// set roll_max to 180, pitch_max to 180, head_max to 180
	p_->roll_min  = -180.0; p_->roll_max  = +180.0;
	p_->pitch_min = -180.0; p_->pitch_max = +180.0;
	p_->head_min  = -180.0; p_->head_max  = +180.0;

	// set roll_speed, pitch_speed, head_speed to 0.1	
	p_->roll_speed  = p_->pitch_speed = p_->head_speed  = 0.1;
	p_->roll_index  = p_->pitch_index = p_->head_index  = 0.0;
}

void pathRandom::message(const ygMessage& msg)
{
	//begin random movement
	if( msg == "start" )
	{
		p_->active = true;
		interpolate(0);	
	}
	//stop random movement
	else if( msg == "stop" )
	{
		p_->active = false;
	}
	//set the speed of the random movement between 0 and 1
	else if( msg == "speed" )
	{
		float speed = msg.floatArg(0);
		
		if( (speed > 1.0) || (speed < 0.0) )
		{
			fprintf(stderr, "ERROR: pathRandom::setSpeed() -> 'speed' must be between 0.0 and 1.0 and request is %f\n", speed );
			fprintf(stderr, "Keeping old 'speed' value of %f\n", p_->speed );
			return;
		}
		
		p_->speed = speed;
	}	
	//set absolute box coordinates
	else  if( msg == "box" )
	{
		pfVec3 min,max;
		
		msg.getVec3Args(min,0);
		msg.getVec3Args(max,3);
	
	    p_->minX = min[0];
		p_->maxX = max[0];
		
		p_->minY = min[1];
		p_->maxY = max[1];
		
		p_->minZ = min[2];
		p_->maxZ = max[2];
		
		setInitPos( );

		loadPath( p_->random_points );		
	}
	else if( msg == "allrot" )
	{	
		switch( msg.args.size() )
		{
			case 0:
				p_->allrot = true;			
				break;
				
			case 9:
			
				pfVec3 min,max, rot_speed;
		
				msg.getVec3Args(min,  0); // get minimun
				msg.getVec3Args(max,  3); // get maximun
				msg.getVec3Args(rot_speed,6); // get rot speed
			
				p_->roll_min  = min[0];
				p_->roll_max  = max[0];
				
				p_->pitch_min = min[1];
				p_->pitch_max = max[1];
				
				p_->head_min  = min[2];
				p_->head_max  = max[2];
				
				p_->roll_speed  = rot_speed[0];
				p_->pitch_speed = rot_speed[1];
				p_->head_speed  = rot_speed[2];  

				p_->allrot = true;
				break;
			
			deafult:
			
				msg.error(name(),"(wrong number of arguments)");
				break;
		}
	}
	else if( msg == "randompoints" )
	{
		p_->random_points = msg.intArg(0);

		loadPath( p_->random_points );		
	}
	else
		ygTransform::message(msg);
}

void pathRandom::app(void)
{
	if ((p_->active) )
	{			
		p_->index = p_->index + p_->speed;
		
		if( (p_->index > 1.0) || (p_->index < 0.0))
		{
			p_->speed = -(p_->speed); // invert increment
			p_->index = p_->index + p_->speed; // return inside limits
		}
		
		// Now the random rotation stuff
		//
		if( p_->allrot )
		{
			p_->roll_index = p_->roll_index + p_->roll_speed;
		
			if( (p_->roll_index > p_->roll_max) || (p_->roll_index < p_->roll_min))
			{
				p_->roll_speed = -(p_->roll_speed); // invert increment
				p_->roll_index = p_->roll_index + p_->roll_speed; // return inside limits
			}

			p_->pitch_index = p_->pitch_index + p_->pitch_speed;
		
			if( (p_->pitch_index > p_->pitch_max) || (p_->pitch_index < p_->pitch_min))
			{
				p_->pitch_speed = -(p_->pitch_speed); // invert increment
				p_->pitch_index = p_->pitch_index + p_->pitch_speed; // return inside limits
			}

			p_->head_index = p_->head_index + p_->head_speed;
		
			if( (p_->head_index > p_->head_max) || (p_->head_index < p_->head_min))
			{
				p_->head_speed = -(p_->head_speed); // invert increment
				p_->head_index = p_->head_index + p_->head_speed; // return inside limits
			}
		}
		
		// Calculate new position and orientation
		interpolate( p_->index );
		
		// send an event with this new 
		// position information
		char str[100];
		
		sprintf(str, "X=%f Y=%f Z=%f ",p_->pos[0], p_->pos[1], p_->pos[2] );	
		
		ygString args( str );

		eventOccurred("pathRandomPos", args);
	}

	ygTransform::app();
}

//
// Local implementation or non-standard class funcion implementations
//

void pathRandom::setInitPos( )
{
	p_->init_x = p_->minX + (p_->maxX - p_->minX)/2.0;
	p_->init_y = p_->minY + (p_->maxY - p_->minY)/2.0;
	p_->init_z = p_->minZ + (p_->maxZ - p_->minZ)/2.0;
	
#ifdef DEBUG
	printf("\nINIT OBJECT POSITION pathRandom: %4.2f %4.2f %4.2f\n",p_->init_x, p_->init_y, p_->init_z );
	printf("\nMIN/MAX POSITION pathRandom min(X Y Z) max(X Y Z): (%4.2f %4.2f %4.2f) (%4.2f %4.2f %4.2f)\n",
	         p_->minX, p_->minY, p_->minZ, p_->maxX, p_->maxY, p_->maxZ);	
#endif
}

void pathRandom::loadPath( unsigned int random_points )
{
#ifdef DEBUG
	printf("\nLOAD PATH pathRandom points: %d\n", random_points );
#endif

	p_->points.clear();

#ifdef DEBUG
	printf("points: ");
#endif
	
	// First point is the initial position
	pfVec3 *v = new pfVec3;
	
	v->vec[0] = p_->init_x;
	v->vec[1] = p_->init_y;
	v->vec[2] = p_->init_z;

	p_->points.push_back(v);
	
	// Then, generate random points and add them	
	while( random_points-- )
	{
		pfVec3 *v = new pfVec3;
		
		v->vec[0] = randomVal( p_->minX,  p_->maxX ); // get a random number between minX and maxX
		v->vec[1] = randomVal( p_->minY,  p_->maxY );
		v->vec[2] = randomVal( p_->minZ,  p_->maxZ );

#ifdef DEBUG
	printf("%4.2f %4.2f %4.2f\n", v->vec[0], v->vec[1], v->vec[2] );
#endif
		
		p_->points.push_back(v);
	}
	
#ifdef DEBUG
	printf("\n");
#endif
}

void pathRandom::interpolate(float t)
{
#ifdef DEBUG
	printf("\nINTERPOLATE pathRandom with index = %f\n", t);
#endif

	float fIndex, frac, heading;
	int index, numPoints = p_->points.size();
	
	fIndex = t * (numPoints-1);
	index = (int)(fIndex);
	frac = fIndex - index;
	
	if (index < 0)
	{
		index = 0;
		frac = 0;
	}
	else if (index > numPoints-2)
	{
		index = numPoints-2;
		frac = 1;
	}
	
	pathRandomCatmullRomCurve curve;
	
	pfVec3 cpt0, cpt1, cpt2, cpt3;
	
	cpt1 = *p_->points[index];
	cpt2 = *p_->points[index+1];
	
	if (index > 0)
		cpt0 = *p_->points[index-1];
	else
		cpt0 = cpt1;
	
	if (index < numPoints-2)
		cpt3 = *p_->points[index+2];
	else
		cpt3 = cpt2;
	
	curve.setControlPoints( cpt0, cpt1, cpt2, cpt3);
	
	p_->pos = curve.value(frac);
	setPosition(p_->pos);

	if( p_->allrot )
	{
		setOrientation(p_->roll_index, p_->pitch_index, p_->head_index);
	}
	else
	{
		p_->dir = curve.tangent(frac);
		heading = pfArcTan2( -(p_->dir[0]), p_->dir[1] );	
		setOrientation(0,0,heading);
	}
}

pathRandomCatmullRomCurve::pathRandomCatmullRomCurve(void)
{
	cpt_[0].set( 0, 0, 0 );
	cpt_[1].set( 0, 0, 0 );
	cpt_[2].set( 0, 0, 0 );
	cpt_[3].set( 0, 0, 0 );

	coeff_[0].set( 0, 0, 0, 0 );
	coeff_[1].set( 0, 0, 0, 0 );
	coeff_[2].set( 0, 0, 0, 0 );
}

void pathRandomCatmullRomCurve::setControlPoints(pfVec3 p0,pfVec3 p1,pfVec3 p2,pfVec3 p3)
{
	cpt_[0] = p0;
	cpt_[1] = p1;
	cpt_[2] = p2;
	cpt_[3] = p3;

	computeCoefficients();
}

pfVec3 pathRandomCatmullRomCurve::value(float t)
{
	pfVec4 T;
	pfVec3 val;

	T.set( t*t*t, t*t, t, 1.0f );

	val[0] = T.dot( coeff_[0] );
	val[1] = T.dot( coeff_[1] );
	val[2] = T.dot( coeff_[2] );

	return val;
}

pfVec3 pathRandomCatmullRomCurve::tangent( float t )
{
	pfVec4 T;
	pfVec3 val;

	T.set( 3.0f*t*t, 2.0f*t, 1.0f, 0.0f );

	val[0] = T.dot( coeff_[0] );
	val[1] = T.dot( coeff_[1] );
	val[2] = T.dot( coeff_[2] );

	return val;
}

void pathRandomCatmullRomCurve::computeCoefficients( void )
{
	pfMatrix CatmullRom (  -0.5,  1.0, -0.5,  0.0,
							1.5, -2.5,  0.0,  1.0,
						   -1.5,  2.0,  0.5,  0.0,
							0.5, -0.5,  0.0,  0.0 );
	pfVec4 geom;
	
	geom.set(cpt_[0][0], cpt_[1][0], cpt_[2][0], cpt_[3][0]);
	coeff_[0].xform( geom, CatmullRom );
	
	geom.set(cpt_[0][1], cpt_[1][1], cpt_[2][1], cpt_[3][1]);
	coeff_[1].xform( geom, CatmullRom );
	
	geom.set(cpt_[0][2], cpt_[1][2], cpt_[2][2], cpt_[3][2]);
	coeff_[2].xform( geom, CatmullRom );
}
