// Description: a transform that moves along a path of points over a duration
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> the point file can be any number of X Y Z corrdinate triplets separated by spaces
//<li> the heading at a point is along the line connecting the next and last points
//<li> this node can also be implemented with a <a href="pointFollower.html">pointFollower</a> node and a <a href="timer.html">timer</a>
//</ul>
//
// Category: Deprecated
// Author: Dave Pape
// Revision: 11/01/01
//
#include <vector>
#include <ygUtil.h>
#include <ygWorld.h>
#include "pathFollower.h"

using namespace std;

extern "C" ygNode* construct_pathFollower(const char* name,bool master) { return new pathFollower(name,master); }


struct _pathFollowerPrivateData
	{
	vector<pfVec3*> points;
	bool active;
	float startTime;
	float duration;
	};


class pathFollowerCatmullRomCurve
	{
	public:
	 pathFollowerCatmullRomCurve(void);
	 void setControlPoints(pfVec3,pfVec3,pfVec3,pfVec3);
	 pfVec3 value(float t);
	 pfVec3 tangent(float t);
	protected:
	 void computeCoefficients(void);
	 pfVec3 cpt_[4];
	 pfVec4 coeff_[3];
	};


pathFollower::pathFollower(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	setClassName("pathFollower");
	p_ = new struct _pathFollowerPrivateData;
	p_->active = false;
	p_->duration = 1.0;
	}


pathFollower::~pathFollower(void)
	{
	delete p_;
	}


void pathFollower::reset(void)
	{
	//set to inactive
	p_->active = false;
	//set the duration to 1.0
	p_->duration = 1.0;
	//clear the points list
	p_->points.clear();
	ygSimpleTransform::reset();
	}


void pathFollower::message(const ygMessage& msg)
	{
	//load a point file
	if (msg == "path")
		loadPath(msg.args[0]);
	//set the path duration
	else if (msg == "duration")
		setDuration(msg.floatArg(0));
	//start moving
	else if (msg == "start")
		start();
	//stop moving
	else if (msg == "stop")
		stop();
	else
		ygSimpleTransform::message(msg);
	}


void pathFollower::loadPath(const ygString& file)
	{
	char *searchPath = ".", line[512];
	ygString fullPath;
	FILE *fp=NULL;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: pathFollower::loadPath: could not find file '%s'\n",
			file.c_str());
		return;
		}
	fp = fopen(fullPath.c_str(),"r");
	if (!fp)
		{
		fprintf(stderr,"ERROR: pathFollower::loadPath could not open \"%s\"\n",
			fullPath.c_str());
		perror(fullPath.c_str());
		return;
		}
	p_->points.clear();
	while (fgets(line,sizeof(line),fp))
		{
		pfVec3 *v = new pfVec3;
		sscanf(line,"%f%f%f",&v->vec[0],&v->vec[1],&v->vec[2]);
		p_->points.push_back(v);
		}
	fclose(fp);
	}


void pathFollower::setDuration(float d)
	{
	if (d < 0.000001)
		d = 1;
	p_->duration = d;
	}


void pathFollower::start(void)
	{
	p_->active = true;
	p_->startTime = ygWorld::FrameTime;
	interpolate(0);
	}


void pathFollower::stop(void)
	{
	p_->active = false;
	}


void pathFollower::app(void)
	{
	if ((p_->active) && (p_->points.size() > 0))
		{
		float t = (ygWorld::FrameTime - p_->startTime) / p_->duration;
		if (t > 1.0)
			t = 1.0;
		interpolate(t);
		if (t >= 1.0)
			stop();
		}
	ygSimpleTransform::app();
	}


void pathFollower::interpolate(float t)
	{
	pfVec3 pos, dir;
	float fIndex, frac, heading;
	int index, numPoints = p_->points.size();
	fIndex = t * (numPoints-1);
	index = (int)(fIndex);
	frac = fIndex - index;
	if (index < 0)
		{
		index = 0;
		frac = 0;
		}
	else if (index > numPoints-2)
		{
		index = numPoints-2;
		frac = 1;
		}
	pathFollowerCatmullRomCurve curve;
	pfVec3 cpt0,cpt1,cpt2,cpt3;
	cpt1 = *p_->points[index];
	cpt2 = *p_->points[index+1];
	if (index > 0)
		cpt0 = *p_->points[index-1];
	else
		cpt0 = cpt1;
	if (index < numPoints-2)
		cpt3 = *p_->points[index+2];
	else
		cpt3 = cpt2;
	curve.setControlPoints(cpt0,cpt1,cpt2,cpt3);
	pos = curve.value(frac);
	dir = curve.tangent(frac);
	heading = pfArcTan2(-dir[0],dir[1]);
	setPosition(pos);
	setOrientation(0,0,heading);
	}




pathFollowerCatmullRomCurve::pathFollowerCatmullRomCurve(void)
	{
	cpt_[0].set(0,0,0);
	cpt_[1].set(0,0,0);
	cpt_[2].set(0,0,0);
	cpt_[3].set(0,0,0);
	coeff_[0].set(0,0,0,0);
	coeff_[1].set(0,0,0,0);
	coeff_[2].set(0,0,0,0);
	}

void pathFollowerCatmullRomCurve::setControlPoints(pfVec3 p0,pfVec3 p1,pfVec3 p2,pfVec3 p3)
	{
	cpt_[0] = p0;
	cpt_[1] = p1;
	cpt_[2] = p2;
	cpt_[3] = p3;
	computeCoefficients();
	}

pfVec3 pathFollowerCatmullRomCurve::value(float t)
	{
	pfVec4 T;
	pfVec3 val;
	T.set(t*t*t,t*t,t,1.0f);
	val[0] = T.dot(coeff_[0]);
	val[1] = T.dot(coeff_[1]);
	val[2] = T.dot(coeff_[2]);
	return val;
	}

pfVec3 pathFollowerCatmullRomCurve::tangent(float t)
	{
	pfVec4 T;
	pfVec3 val;
	T.set(3.0f*t*t,2.0f*t,1.0f,0.0f);
	val[0] = T.dot(coeff_[0]);
	val[1] = T.dot(coeff_[1]);
	val[2] = T.dot(coeff_[2]);
	return val;
	}

void pathFollowerCatmullRomCurve::computeCoefficients(void)
	{
	pfMatrix CatmullRom (  -0.5,  1.0, -0.5,  0.0,
				 1.5, -2.5,  0.0,  1.0,
				-1.5,  2.0,  0.5,  0.0,
				 0.5, -0.5,  0.0,  0.0 );
	pfVec4 geom;
	geom.set(cpt_[0][0], cpt_[1][0], cpt_[2][0], cpt_[3][0]);
	coeff_[0].xform(geom,CatmullRom);
	geom.set(cpt_[0][1], cpt_[1][1], cpt_[2][1], cpt_[3][1]);
	coeff_[1].xform(geom,CatmullRom);
	geom.set(cpt_[0][2], cpt_[1][2], cpt_[2][2], cpt_[3][2]);
	coeff_[2].xform(geom,CatmullRom);
	}
