// Description: a space that generates events in the presence of a node
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node generates events with the name of the node detected
//<li> the local mode determines if the node position is local to the object or relative to world coordinates
//</ul>
//
// Category: Trigger
// Author: Alex Hill
// Revision: 10/01/02
//
#include <set>
#include <ygNodeDB.h>
#include <ygUtil.h>
#include <ygWorld.h>
#include "nodeTrigger.h"

using namespace std;

extern "C" ygNode* construct_nodeTrigger(const char* name,bool master) { return new nodeTrigger(name,master); }

struct _nodeTriggerPrivateData
	{
	ygString detectType;
	set<ygNode*> prevInside;
	bool debugInside;
	int lastFrame;
	};


nodeTrigger::nodeTrigger(const char* name,bool master) : ygSpace(name,master)
	{
	setClassName("nodeTrigger");
	p_ = new struct _nodeTriggerPrivateData;
	p_->lastFrame = 0;
	debugFlag("inside", &p_->debugInside);
	}

nodeTrigger::~nodeTrigger(void)
	{
	delete p_;
	}


void nodeTrigger::reset(void)
	{
	//clear set of nodes previously inside
	p_->prevInside.clear();
	p_->lastFrame = ygWorld::FrameNumber;
	ygSpace::reset();
	}


void nodeTrigger::message(const ygMessage& msg)
	{
	//set the type of node to detect
	if (msg == "detect")
		{
		if (msg.args.size() > 0)
			setDetectType(msg.args[0]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSpace::message(msg);
	}


void nodeTrigger::setDetectType(const ygString& s)
	{
	p_->detectType = s;
	}


void nodeTrigger::app(void)
	{
	set<ygNode*> inside;
	int i;
	if (p_->lastFrame < (ygWorld::FrameNumber-1))
		p_->prevInside.clear();
	p_->lastFrame = ygWorld::FrameNumber;
	const vector<ygNode*>& nodes = ygNodeDB::nodesOfType(p_->detectType);
	//for each node of detect type in the world
	for (i=0; i < nodes.size(); i++)
		{
		//if node is within the space
		if (contains(nodes[i]->origin(this)))
			{
			if (p_->debugInside)
				cout << name() << ": " << nodes[i]->name() << " is inside\n";
			//add to set of nodes inside
			inside.insert(inside.begin(),nodes[i]);
			//if inside but not previously then generate enter event
			if (p_->prevInside.find(nodes[i]) == p_->prevInside.end())
				{
				ygString args("node=");
				args += nodes[i]->name();
				//the node has entered the space
				eventOccurred("enter",args);
				}
			}
		}
	set<ygNode*>::const_iterator iter;
	//for each node previously inside
	for (iter = p_->prevInside.begin(); iter != p_->prevInside.end();
	     ++iter)
		//if not currently inside then generate exit event
		if (inside.find(*iter) == inside.end())
			{
			ygString args("node=");
			args += (*iter)->name();
			//the node has exited the space
			eventOccurred("exit",args);
			}
	p_->prevInside = inside;
	ygSpace::app();
	}
