// Description: a space that generates position events in the presence of a node type
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node generates events with the name of the node detected
//</ul>
//
// Category: Trigger
// Author: Dave Pape
// Revision: 11/01/01
//
#include <ygNodeDB.h>
#include <ygUtil.h>
#include <ygWorld.h>
#include "nodePosition.h"

using namespace std;

extern "C" ygNode* construct_nodePosition(const char* name,bool master) { return new nodePosition(name,master); }


nodePosition::nodePosition(const char* name,bool master) : ygSpace(name,master)
	{
	setClassName("nodePosition");
	debugFlag("inside",&debugInside);
	}

nodePosition::~nodePosition(void)
	{
	}


void nodePosition::reset(void)
	{
	//clear set of nodes previously inside
	prevInside.clear();
	//set reporting mode to local
	local = true;
	lastFrame = ygWorld::FrameNumber;
	ygSpace::reset();
	}


void nodePosition::message(const ygMessage& msg)
	{
	//set the type of node to detect
	if (msg == "detect")
		{
		if (msg.args.size() > 0)
			setDetectType(msg.args[0]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set position reporting mode
	else if (msg == "local")
		{
		if (msg.args.size() > 0)
			local = msg.boolArg(0);
		else
			local = true;
		}
	else
		ygSpace::message(msg);
	}


void nodePosition::setDetectType(const ygString& s)
	{
	detectType = s;
	}


void nodePosition::app(void)
	{
	set<ygNode*> inside;
	int i;
	if (lastFrame < (ygWorld::FrameNumber-1))
		prevInside.clear();
	lastFrame = ygWorld::FrameNumber;
	const vector<ygNode*>& nodes = ygNodeDB::nodesOfType(detectType);
	//for each node of detect type in the world
	for (i=0; i < nodes.size(); i++)
		{
		//if node is within the space
		pfVec3 position = nodes[i]->origin(this);
		if (contains(position))
			{
			if (debugInside)
				cout << name() << ": " << nodes[i]->name() << " is inside\n";
			//add to set of nodes inside
			inside.insert(inside.begin(),nodes[i]);
			ygString args("node=");
			args += nodes[i]->name();
			if (!local)
				position = nodes[i]->origin();
			char value[32];
			args += " xpos=";
			sprintf(value,"%f",position[0]);
			args += value;
			args += " ypos=";
			sprintf(value,"%f",position[1]);
			args += value;
			args += " zpos=";
			sprintf(value,"%f",position[2]);
			args += value;
			//a node is inside the space
			eventOccurred("inside",args);
			//if inside but not previously then generate enter event
			if (prevInside.find(nodes[i]) == prevInside.end())
				{
				//the node has entered the space
				eventOccurred("enter",args);
				}
			}
		}
	set<ygNode*>::const_iterator iter;
	//for each node previously inside
	for (iter = prevInside.begin(); iter != prevInside.end();
	     ++iter)
		//if not currently inside then generate exit event
		if (inside.find(*iter) == inside.end())
			{
			ygString args("node=");
			args += (*iter)->name();
			//the node has exited the space
			eventOccurred("exit",args);
			}
	prevInside = inside;
	ygSpace::app();
	}
