//Description: plays a movie on a rectangular surface, using xine 
//
//<b>notes:</b>
//<ul>
//<li> this node currently is supported only on LINUX machines, using the xine library. 
//<li> information about xine, can be found <a href="http://www.evl.uic.edu/eleni/yg/xine.html"> here.</a>
//<li> information regarding the production path of movies that have been tested, can be found <a href="http://www.evl.uic.edu/eleni/yg/movie_production.html"> here.</a>
//<li> an example movie file, can be found <a href="http://www.evl.uic.edu/eleni/yg/downloads/PigeonsSF.mov">here</a>.
//<li> the following formats are currently examined to be supported:
//<ul>
//<li> MPEG1 (*.mpg)
//<li> MPEG2 (*.mpg)
//<li> MPEG4 (*.mpg)
//<li> WINDOWS MEDIA (*.wmv)
//<li> REAL MEDIA (*.rm) - win32 codecs are necessary.For further information please click <a href="http://www.evl.uic.edu/eleni/yg/rm.html"> here.</a>
//<li> QUICKTIME (*.mov)
//</ul>
//</ul>
//
// Category: Attributes
// Author: Kalle Jalkanen and Jonas Westling, Interactive Institute Ume, Sweden
//           02/24/03
// Revision: 08/10/03 Helen-Nicole Kostis - documentation, examination of more movie formats and production path of them
//
//
#include <CAVERNnet_udp_c.hxx>
#include <CAVERN.hxx>
#include <errno.h>
#include "movieSurface.h"
#include "MoviePlayer.h"

#define MAX_IMG 	29
#define ANIM_HZ 	15

using namespace std;


extern "C" ygNode* construct_movieSurface(const char* name,bool master) 
{ 
	return new movieSurface(name,master); 
}


struct _movieSurfacePrivateData
{
	float frameTime;	
	MoviePlayer *mp;
	int status;
	bool playing;
	ygString movieFile;
	
};



movieSurface::movieSurface(const char* name, bool master) : rectSurface(name, master)
{
	setClassName("movieSurface", true); 
	_p = new struct _movieSurfacePrivateData;
	_p->mp = new MoviePlayer();
	_p->status = 0;
	_p->movieFile = "";
	addNetKey("file", &_p->movieFile, YG_NET_STRING);	
	_p->playing = false;
	addNetKey("playing", &_p->playing, YG_NET_BOOL);

}


movieSurface::~movieSurface()
{
	stopMovie();	
	delete _p->mp;
   	delete _p;
}


void movieSurface::reset()
{
	rectSurface::reset();
}



void movieSurface::stopMovie()
{
	if (_p->status) 
	{
		rectSurface::clear(0L);
		rectSurface::setImg(NULL, 0, 0, 0);
		_p->mp->stop();
		_p->mp->close();			
	
		if (isNetMaster()) 
		{
			eventOccurred("end");
		}
	}
	_p->status = 0;

}


void movieSurface::playMovie()
{
	ygString _movieFile("");
	if (_p->movieFile == "") return;

	char *mf = getenv("MOVIE_PATH");	
	if (mf) 
	{
		_movieFile.append( mf );
		if (_movieFile[_movieFile.length()-1] != '/')
		{
			_movieFile.append("/");
		}
	}

	_movieFile += _p->movieFile;
	
	if ( _p->mp->open( _movieFile.c_str() ) ) 
	{
		if ( _p->mp->play(0,0) ) 
			_p->status = 1;
		else 
			cout<<"Unable to play movie\n";
		
	} 
	else 
	{
		cout << "Unable to open movie file: '" << _movieFile.c_str() <<"'"<< endl;
	}
}



void movieSurface::message(const ygMessage& msg)
{
	if (msg == "play") {

		stopMovie();
		
		_p->playing = true;
		netKeyChanged("playing");	

		playMovie();		

	} else if (msg == "stop") {
		_p->playing = false;
		netKeyChanged("playing");	
	
		stopMovie();	
		
	} else if (msg == "file") {
		
		if ( msg.args.size() != 1 ) {
			msg.error("","(wrong number of arguments - should be 'file(filename)' )");
		} else {
			_p->movieFile = msg.stringArg(0);
			netKeyChanged("file");
		}
	} else {
		rectSurface::message(msg);
	}
}




void movieSurface::app(void)
{
	
	if ( rectSurface::isCreated() ) 
	{
		if (_p->mp->app())
		{
			rectSurface::setImg(_p->mp->getFrame(), _p->mp->getWidth(), _p->mp->getHeight(), 3);
		} 
		else 
		{
			stopMovie();
		}	

	}

	rectSurface::app();
}


void movieSurface::acceptNetKey(const ygString& key)
{
	if (key == "file") 
	{
			
	} 
	else if (key == "playing") 
	{
		stopMovie();
		
		if (_p->playing) 
		{
			playMovie();
		} 
	
	} 
	
	else
		rectSurface::acceptNetKey(key);
}





