// Description: a quaternian based motion path between two positions and orientaions
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the set/value/position message takes a value between 0 and 1
//<li> values outside of the 0 to 1 range will be interpolated accordingly
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/01
//
#include <ygNodeDB.h>
#include <ygWorld.h>
#include "mover.h"

extern "C" ygNode* construct_mover(const char* name,bool master) { return new mover(name,master); }

mover::mover(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	setClassName("mover");
	position = 0.0;
	posQuat.pos.set(0, 0, 0);
	posQuat.quat.set(0, 0, 0, 1);
	startPosQuat.pos.set(0, 0, 0);
	startPosQuat.quat.set(0, 0, 0, 1);	
	endPosQuat.pos.set(0, 0, 0);  
	endPosQuat.quat.set(0, 0, 0, 1);
	}

mover::~mover(void)
	{
	}

void mover::reset(void)
	{
	//set current position to zero
	position = 0.0;
	posQuat.pos.set(0, 0, 0);
	posQuat.quat.set(0, 0, 0, 1);
	//set start position and orientation to (0,0,0)
	startPosQuat.pos.set(0, 0, 0);
	startPosQuat.quat.set(0, 0, 0, 1);	
	//set end position and orientation to (0,0,0)
	endPosQuat.pos.set(0, 0, 0);  
	endPosQuat.quat.set(0, 0, 0, 1);
	ygSimpleTransform::reset();
	}

void mover::message(const ygMessage& msg)
	{
	//set the start position of the path
	if (msg == "startPosition")
		{
		pfVec3 pos;
		msg.getVec3Args(pos);
		setStartPosition(pos);
		}
	//set the start orientation of the path
	else if (msg == "startOrientation")
		{
		pfVec3 rot;
		msg.getVec3Args(rot);
		setStartOrientation(rot);
		}
	//set the end position of the path
	else if (msg == "endPosition")
		{
		pfVec3 pos;
		msg.getVec3Args(pos);
		setEndPosition(pos);
		}
	//set the end orientation of the path
	else if (msg == "endOrientation")
		{
		pfVec3 rot;
		msg.getVec3Args(rot);
		setEndOrientation(rot);
		}
	//set the position along the motion path
	else if (msg == "set" || msg == "value" || msg == "position")
		{
		position = msg.floatArg(0);
		}
	//set the position to the start position
	else if (msg == "reset")
		{
		pfMatrix mat;
		mat.makeQuat(startPosQuat.quat);
		mat.postTrans(mat,startPosQuat.pos[0],startPosQuat.pos[1],startPosQuat.pos[2]);
		setMatrix(mat);
		}
	else
		ygSimpleTransform::message(msg);
	}
	
void mover::setStartPosition(const pfVec3& pos)
	{
	startPosQuat.pos = pos;
	}
		
void mover::setStartOrientation(const pfVec3& rot)
	{
	pfMatrix mat;
	mat.makeEuler(rot[2],rot[0],rot[1]);
	mat.getOrthoQuat(startPosQuat.quat);
	}
	
void mover::setEndPosition(const pfVec3& pos)
	{
	endPosQuat.pos = pos;
	}

void mover::setEndOrientation(const pfVec3& rot)
	{
	pfMatrix mat;
	mat.makeEuler(rot[2],rot[0],rot[1]);
	mat.getOrthoQuat(endPosQuat.quat);
	}

void mover::app(void)
	{
	pfMatrix mat;
	getMatrix(mat);
	pfCoord coord;
	mat.getOrthoCoord(&coord);
	posQuat.pos = coord.xyz;
	mat.getOrthoQuat(posQuat.quat);
	//interpolate the position
	posQuat.pos = startPosQuat.pos + position *
			(endPosQuat.pos - startPosQuat.pos);
	//interpolate the orientation with quaternians
	posQuat.quat.slerp(position,startPosQuat.quat,endPosQuat.quat);
	mat.makeQuat(posQuat.quat);
	mat.postTrans(mat,posQuat.pos[0],posQuat.pos[1],posQuat.pos[2]);
	//set the matrix
	setMatrix(mat);
	ygSimpleTransform::app();
	}
