// Description: creates a morphed object between two or more models
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> models can have diferent textures but must have identical primitive format
//<li> multiple models can be given to generate multiple keyframes
//<li> keyframes are numbered beginning with 0 up to the number of models minus 2
//<li> the set/value/position message takes a value between 0 and 1
//<li> the set/value/position message is useful for moving smoothly between all models
//<li> the keyframe message takes the keyframe number and a decimal percentage into the keyframe
//<li> the keyframe message is useful for moving between keyframes at different rates
//</ul>
//
// Category: Geometry
// Author: Alex Hill
// Revision: 04/01/04 Alex Hill - added pfdBldrMode call to disable mesh building
//           11/01/01
//
#include <Performer/pr.h>
#include <Performer/pfdu.h>
#include <Performer/pr/pfList.h>
#include <Performer/pf/pfChannel.h>
#include <Performer/pr/pfGeoState.h>
#include <Performer/pr/pfGeoSet.h>
#include <ygPFObjectCache.h>
#include <ygNetKeys.h>
#include <ygWorld.h>
#include "morpher.h"

extern "C" ygNode* construct_morpher(const char* name,bool master) { return new morpher(name,master); }


struct geoSetData
	{
	pfVec3* coords;
	pfVec3* norms;
	pfVec4* colors;
	pfVec2* texCoords;
	pfMaterial* frontMtl;
	pfMaterial* backMtl;
	pfTexture* texture;           
	pfTexEnv* texenv;
	};

struct fluxGeoSetData
	{
	int numColors;
	pfEngine* coordsEngine;
	pfEngine* normsEngine;
	pfEngine* texCoordsEngine;
	pfGeoSet* solidGSet;
	pfGeoSet* transGSet;
	};

morpher::morpher(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("morpher",true);
	//create frame weight flux
	geode = NULL;
	fluxFrame = new pfFlux(sizeof(float),PFFLUX_DEFAULT_NUM_BUFFERS);
	fluxFrame->setMode(PFFLUX_PUSH,PF_ON);
	keyframes = new pfList;
	fluxDataList = new pfList;
	numFrames = 0;
	frame = 0;
	position = 0.0;
	addNetKey("filenames",&filenames,YG_NET_STRING);
	addNetKey("frame",&frame,YG_NET_INT);
	addNetKey("position",&position,YG_NET_FLOAT);
	}

void morpher::reset(void)
	{
	if (geode)
		pfnode()->removeChild(geode);
	geode = NULL;
	//reset the number of frames to 0
	numFrames = 0;
	//reset the keyframe list
	keyframes = new pfList;
	fluxDataList = new pfList;
	fluxFrame = new pfFlux(sizeof(float),PFFLUX_DEFAULT_NUM_BUFFERS);
	fluxFrame->setMode(PFFLUX_PUSH,PF_ON);
	filenames.clear();
	netKeyChanged("filenames");
	frame = 0;
	netKeyChanged("frame");
	position = 0.0;
	netKeyChanged("position");
	ygNode::reset();
	}
	 
void morpher::message(const ygMessage& msg)
	{
	//add a model file to the list of keyframes
	if (msg == "model" || msg == "file")
		{
		ygString newModel = msg.args[0];
		addModel(newModel);
		filenames += newModel;
		filenames += " ";
		netKeyChanged("filenames");
		}
	//select a keyframe and a decimal percentage within
	else if (msg == "keyframe")
		{
		float newFrame = msg.floatArg(0);
		if ((int)newFrame != frame)
			{
			setFrame((int)newFrame);
			netKeyChanged("frame");
			}
		float newPosition = newFrame - (int)newFrame;
		setPosition(newPosition);
		netKeyChanged("position");
		}
	//select a percentage through all keyframes
	else if (msg == "set" || msg == "value" || msg == "position")
		{
		float newValue = msg.floatArg(0);
		int newFrame = (int)(newValue*(numFrames-1));
		float newPosition = newValue*(numFrames-1) - newFrame;
		if (newFrame == numFrames-1)
			{
			newFrame = numFrames - 1;
			newPosition = 1.0;
			}
		if (newFrame != frame)
			{
			setFrame(newFrame);
			netKeyChanged("frame");
			}
		setPosition(newPosition);
		netKeyChanged("position");
		}
	else
		ygNode::message(msg);
	}

void morpher::addModel(const ygString& file)
	{
	pfdBldrMode(PFDBLDR_MESH_ENABLE,0);
	pfdBldrMode(PFDBLDR_MESH_MAX_TRIS,0);
	pfNode *node = pfdLoadFile(file.c_str());
	pfdBldrMode(PFDBLDR_MESH_ENABLE,1);
	pfdBldrMode(PFDBLDR_MESH_MAX_TRIS,0);
	if (node)
		{
		numFrames++;
		if (!geode)
			{
			geode = new pfGeode;
			pfnode()->addChild(geode);
			copyModelToGeode(node);
			geode->setTravMask(PFTRAV_ISECT, YG_ISECT_ALLMASK, PFTRAV_SELF|PFTRAV_DESCEND|PFTRAV_IS_CACHE, PF_SET);
			}
		pfList* geoSetList = new pfList;
		getGeoSetData(node,geoSetList);
		keyframes->add(geoSetList);
		if (numFrames == 2)
			setFrame(0);
		}
	}
    
void morpher::copyModelToGeode(pfNode *node)
	{
	int i;
	if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode*)node;
		for (i=0; i < geode->getNumGSets(); i++)
			copyGeoSet(geode->getGSet(i));
		}
	else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup*)node;
		for (i=0; i < group->getNumChildren(); i++)
			copyModelToGeode(group->getChild(i));
		}
	}

void morpher::copyGeoSet(pfGeoSet* gset)
	{
	int indexSize,maxIndex;
	void* attrList;
	ushort* indexList;
	
	//create new flux geoset data structure
	struct fluxGeoSetData* fluxGSetData;
	fluxGSetData = new struct fluxGeoSetData;
	
	//create solid geoset
	fluxGSetData->solidGSet = new pfGeoSet;
	fluxGSetData->solidGSet->setPrimType(gset->getPrimType());
	fluxGSetData->solidGSet->setNumPrims(gset->getNumPrims());
	fluxGSetData->solidGSet->setPrimLengths(gset->getPrimLengths());
	
	//create and setup new coordinates
	pfFlux* fluxCoords;
	indexSize = gset->getAttrRange(PFGS_COORD3,NULL,&maxIndex);
	indexSize = PF_MAX2(indexSize,maxIndex+1);
	gset->getAttrLists(PFGS_COORD3,&attrList,&indexList);
	fluxCoords = new pfFlux(sizeof(pfVec3)*indexSize,PFFLUX_DEFAULT_NUM_BUFFERS);
	fluxCoords->initData(attrList);
	fluxGSetData->solidGSet->setAttr(PFGS_COORD3,gset->getAttrBind(PFGS_COORD3),fluxCoords,indexList);
	
	//set up coordinate flux engine
	fluxGSetData->coordsEngine = new pfEngine(PFENG_MORPH);
	fluxGSetData->coordsEngine->setSrc(PFENG_MORPH_FRAME,fluxFrame,NULL,0,0,1);
	fluxGSetData->coordsEngine->setDst(fluxCoords,NULL,0,3);
	fluxGSetData->coordsEngine->setIterations(indexSize,3);
	
	//create ans set up new normals
	pfFlux* fluxNorms;
	indexSize = gset->getAttrRange(PFGS_NORMAL3,NULL,&maxIndex);
	indexSize = PF_MAX2(indexSize,maxIndex+1);
	gset->getAttrLists(PFGS_NORMAL3,&attrList,&indexList);
	fluxNorms = new pfFlux(sizeof(pfVec3)*indexSize,PFFLUX_DEFAULT_NUM_BUFFERS);
	fluxNorms->initData(attrList);
	fluxGSetData->solidGSet->setAttr(PFGS_NORMAL3,gset->getAttrBind(PFGS_NORMAL3),fluxNorms,indexList);
	
	//set up normal flux engine
	fluxGSetData->normsEngine = new pfEngine(PFENG_MORPH);
	fluxGSetData->normsEngine->setSrc(PFENG_MORPH_FRAME,fluxFrame,NULL,0,0,1);
	fluxGSetData->normsEngine->setDst(fluxNorms,NULL,0,3);
	fluxGSetData->normsEngine->setIterations(indexSize,3);    
	
	//create new texture coordinates
	pfFlux* fluxTexCoords;
	indexSize = gset->getAttrRange(PFGS_TEXCOORD2, NULL, &maxIndex);
	indexSize = PF_MAX2(indexSize,maxIndex+1);
	gset->getAttrLists(PFGS_TEXCOORD2,&attrList,&indexList);
	fluxTexCoords = new pfFlux(sizeof(pfVec2)*indexSize,PFFLUX_DEFAULT_NUM_BUFFERS);
	fluxTexCoords->initData(attrList);
	fluxGSetData->solidGSet->setAttr(PFGS_TEXCOORD2,gset->getAttrBind(PFGS_TEXCOORD2),fluxTexCoords,indexList);
	
	//set up texture coordinate flux engine
	fluxGSetData->texCoordsEngine = new pfEngine(PFENG_MORPH);
	fluxGSetData->texCoordsEngine->setSrc(PFENG_MORPH_FRAME,fluxFrame,NULL,0,0,1);
	fluxGSetData->texCoordsEngine->setDst(fluxTexCoords,NULL,0,2);
	fluxGSetData->texCoordsEngine->setIterations(indexSize,2);    
	
	//set up color coordinates
	indexSize = gset->getAttrRange(PFGS_COLOR4,NULL,&maxIndex);
	indexSize = PF_MAX2(indexSize,maxIndex+1);
	fluxGSetData->numColors = indexSize;
	gset->getAttrLists(PFGS_COLOR4,&attrList,&indexList);
	fluxGSetData->solidGSet->setAttr(PFGS_COLOR4,gset->getAttrBind(PFGS_COLOR4),attrList,indexList);
	
	//create transparent geoset
	fluxGSetData->transGSet = new pfGeoSet;
	fluxGSetData->transGSet->copy(fluxGSetData->solidGSet);
	
	//get geoState attributes
	pfGeoState* solidGState = new pfGeoState;
	pfGeoState* transGState = new pfGeoState;
	solidGState->copy(gset->getGState());
	transGState->copy(gset->getGState());
	
	//set up transparency
	transGState->setMode(PFSTATE_TRANSPARENCY,PFTR_BLEND_ALPHA);
	fluxGSetData->transGSet->setDrawBin(PFSORT_TRANSP_BIN);    
	
	//add new geoStates to geoSets
	fluxGSetData->solidGSet->setGState(solidGState);
	fluxGSetData->transGSet->setGState(transGState);
	
	//add new geoSets to geodes
	geode->addGSet(fluxGSetData->solidGSet);
	geode->addGSet(fluxGSetData->transGSet);
	
	//add geoSet data to geoSet list
	fluxDataList->add(fluxGSetData);
	}
    
void morpher::getGeoSetData(pfNode* node,pfList* list)
	{
	int i;
	if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode*)node;
		for (i=0;i<geode->getNumGSets();i++)
			{
			struct fluxGeoSetData* fluxGSetData = (struct fluxGeoSetData*)fluxDataList->get(list->getNum());
			struct geoSetData* gSetData = new struct geoSetData;
			pfGeoSet* gset = geode->getGSet(i);
			pfVec3* coords;
			pfVec3* norms;
			pfVec2* texCoords;
			ushort* ilist;
			gset->getAttrLists(PFGS_COORD3,(void**)&coords,&ilist);
			gset->getAttrLists(PFGS_NORMAL3,(void**)&norms,&ilist);
			gset->getAttrLists(PFGS_COLOR4,(void **)&gSetData->colors,&ilist);
			gset->getAttrLists(PFGS_TEXCOORD2,(void**)&texCoords,&ilist);
			fluxGSetData->coordsEngine->setSrc(PFENG_MORPH_SRC(numFrames-1),coords,NULL,0,0,3);
			fluxGSetData->normsEngine->setSrc(PFENG_MORPH_SRC(numFrames-1),norms,NULL,0,0,3);
			fluxGSetData->texCoordsEngine->setSrc(PFENG_MORPH_SRC(numFrames-1),texCoords,NULL,0,0,2);
			pfGeoState* gState = gset->getGState();
			pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
			if (frontMtl)
				{
				gSetData->frontMtl = new pfMaterial;
				gSetData->frontMtl->copy(frontMtl);
				}
			else
				{
				gSetData->frontMtl = NULL;
				}
			pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
			if (backMtl)
				{
				gSetData->backMtl = new pfMaterial;
				gSetData->backMtl->copy(backMtl);
				}
			else
				{
				gSetData->backMtl = NULL;
				}
			gSetData->texture = (pfTexture*)gState->getAttr(PFSTATE_TEXTURE);
			gSetData->texenv = (pfTexEnv*)gState->getAttr(PFSTATE_TEXENV);			
			list->add(gSetData);
			}
		}
	else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		for (i=0;i<group->getNumChildren();i++)
			getGeoSetData(group->getChild(i),list);
		}
	}

void morpher::setPosition(float p)
	{
	if (p > 1.0)
		position = 1.0;
	else if (p < 0.0)
		position = 0.0;
	else
		position = p;
	float* frameWeight = (float*)fluxFrame->getWritableData();
	frameWeight[0] = float(frame)+position;
	fluxFrame->writeComplete();
	}

void morpher::setFrame(int number)
	{
	if (numFrames > 1 && number >= 0 && number < numFrames-1)
		{
		frame = number;
		float* frameWeight = (float*)fluxFrame->getWritableData();
		frameWeight[0] = float(frame)+position;
		fluxFrame->writeComplete();
		pfList* solidGSetDataList = (pfList*)keyframes->get(frame);
		pfList* transGSetDataList = (pfList*)keyframes->get(frame+1);
		for (int i=0;i<fluxDataList->getNum();i++)
			{
			int binding;
			void* attrList;
			ushort* indexList;
			struct fluxGeoSetData* fluxGSetData = (struct fluxGeoSetData*)fluxDataList->get(i);
			struct geoSetData* solidGSetData = (struct geoSetData*)solidGSetDataList->get(i);
			struct geoSetData* transGSetData = (struct geoSetData*)transGSetDataList->get(i);
			
			//retrieve existing index list and binding
			fluxGSetData->solidGSet->getAttrLists(PFGS_COLOR4,&attrList,&indexList);
			binding = fluxGSetData->solidGSet->getAttrBind(PFGS_COLOR4);
			
			//set solid colors and materials
			fluxGSetData->solidGSet->setAttr(PFGS_COLOR4,binding,solidGSetData->colors,indexList);
			fluxGSetData->solidGSet->getGState()->setAttr(PFSTATE_FRONTMTL,solidGSetData->frontMtl);
			fluxGSetData->solidGSet->getGState()->setAttr(PFSTATE_BACKMTL,solidGSetData->backMtl);
			fluxGSetData->solidGSet->getGState()->setAttr(PFSTATE_TEXTURE,solidGSetData->texture);
			fluxGSetData->solidGSet->getGState()->setAttr(PFSTATE_TEXENV,solidGSetData->texenv);
			
			//set transparent colors and materials
			fluxGSetData->transGSet->setAttr(PFGS_COLOR4,binding,transGSetData->colors,indexList);
			fluxGSetData->transGSet->getGState()->setAttr(PFSTATE_FRONTMTL,transGSetData->frontMtl);
			fluxGSetData->transGSet->getGState()->setAttr(PFSTATE_BACKMTL,transGSetData->backMtl);
			fluxGSetData->transGSet->getGState()->setAttr(PFSTATE_TEXTURE,transGSetData->texture);	    
			fluxGSetData->transGSet->getGState()->setAttr(PFSTATE_TEXENV,transGSetData->texenv);	    
			}
		} 
	}

void morpher::app(void)
	{
	if (numFrames > 1)
		{
		pfList* transGSetDataList = (pfList*)keyframes->get(frame+1);
		for (int i=0;i<fluxDataList->getNum();i++)
			{
			struct fluxGeoSetData* fluxGSetData = (struct fluxGeoSetData*)fluxDataList->get(i);
			struct geoSetData* transGSetData = (struct geoSetData*)transGSetDataList->get(i);
			if (transGSetData->frontMtl)
				transGSetData->frontMtl->setAlpha(position);
			if (transGSetData->backMtl)
				transGSetData->backMtl->setAlpha(position);
			for (int j=0;j<fluxGSetData->numColors;j++)
				transGSetData->colors[j][3] = position;
			}
		updateBounds();
		}
	ygNode::app();
	}

void morpher::updateBounds(void)
	{
	for (int i=0;i<geode->getNumGSets();i++)
		{
		geode->getGSet(i)->setBound(NULL,PFBOUND_DYNAMIC);
		}
	pfNode* node = geode;
	while (node)
		{
		node->setBound(NULL,PFBOUND_DYNAMIC);
		if (node->getNumParents())
			node = node->getParent(0);
		else
			node = NULL;
		}
	}

void morpher::acceptNetKey(const ygString& key)
	{
	if (key == "filenames")
		{
		if (filenames.length() > 0)
			{
			ygString* currName;
			int i = 0;
			int currFrame = 0;
			while (currName = filenames.nextToken(" ",&i))
				{
				currFrame++;
				if (currFrame > numFrames)
					addModel(*currName);
				delete currName;
				}
			}
		else
			{
			if (geode)
				pfnode()->removeChild(geode);
			geode = NULL;
			numFrames = 0;
			keyframes = new pfList;
			fluxDataList = new pfList;
			fluxFrame = new pfFlux(sizeof(float),PFFLUX_DEFAULT_NUM_BUFFERS);
			fluxFrame->setMode(PFFLUX_PUSH,PF_ON);
			}
		}		
	else if (key == "frame")
		setFrame(frame);
	else if (key == "position")
		setPosition(position);
	else
		ygNode::acceptNetKey(key);
    }
