// Description: a space within which all subject geometry and users are mirrored
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node currently does not support arbitrary orientation and should not be rotated
//<li> you must add the following to your RUN file in order to use this node:<br>
//<ul>
//<li> setenv YG_PRELOAD_CLASSES mirror
//</ul>
//</ul>
//
// Category: Geometry
// Author: Alex Hill
// Revision: 11/10/01
//
#include <ygUser.h>
#include <ygWorld.h>
#include <ygUtil.h>
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include "mirror.h"


extern "C" ygNode* construct_mirror(const char* name,bool master) { return new mirror(name,master); }


mirror::mirror(const char* name,bool master) : ygSpace(name,master)
	{
	setClassName("mirror",true);
	pfnode()->setTravFuncs(PFTRAV_DRAW,preDrawFunc,postDrawFunc);
	dcsUsers = new pfDCS;
	pfnode()->addChild(dcsUsers);
	reflection.makeIdent();
	reflection.setRow(1,0,-1,0,0);
	axis = 1;
	//distribute the list of subjects
	addNetKey("subject",&subjectList,YG_NET_STRING);
	//distribute the mirroring axis
	addNetKey("axis",&axis,YG_NET_INT);
	}

mirror::~mirror(void)
	{
	}

void mirror::message(const ygMessage& msg)
	{
	//add a subject node
	if (msg == "subject")
		{
		ygString newSubject = msg.args[0];
		addSubject(newSubject);
		subjectList += newSubject;
		subjectList += " ";
		netKeyChanged("subject");
		}
	//set the axis for mirroring
	else if (msg == "axis")
		{
		if (msg.args[0] == "x")
			axis = 0;
		else if (msg.args[0] == "y")
			axis = 1;
		else if (msg.args[0] == "z")
			axis = 2;
		else if (msg.args[0] == "xy")
			axis = 3;
		setAxis();
		netKeyChanged("axis");
		}
	else
		ygSpace::message(msg);
	}

void mirror::reset(void)
	{
	//reset the list of subjects
	prevInside.clear();
	subjects.clear();
	subjectList.clear();
	ygSpace::reset();
	}

int mirror::preDrawFunc(pfTraverser* trav,void* data)
	{
	pfPushState();
	pfCullFace(PFCF_OFF);
	pfOverride(PFSTATE_CULLFACE,PF_ON);
	return PFTRAV_CONT;
	}
	
int mirror::postDrawFunc(pfTraverser* trav,void* data)
	{
	pfOverride(PFSTATE_CULLFACE,PF_OFF);
	pfPopState();
	return PFTRAV_CONT;
	}

void mirror::setAxis(void)
	{
	if (axis == 0)
		{
		reflection.setRow(0,-1,0,0,0);
		reflection.setRow(1,0,1,0,0);
		reflection.setRow(2,0,0,1,0);
		}
	else if (axis == 1)
		{
		reflection.setRow(0,1,0,0,0);
		reflection.setRow(1,0,-1,0,0);
		reflection.setRow(2,0,0,1,0);
		}
	else if (axis == 2)
		{
		reflection.setRow(0,1,0,0,0);
		reflection.setRow(1,0,1,0,0);
		reflection.setRow(2,0,0,-1,0);
		}
	else if (axis == 3)
		{
		reflection.setRow(0,-1,0,0,0);
		reflection.setRow(1,0,-1,0,0);
		reflection.setRow(2,0,0,1,0);
		}
	}
	
void mirror::app(void)
	{
	set<ygUser*> inside;
	int i;
	//for each user in the world
	for (i=0; i < ygWorld::World->numUsers(); i++)
		{
		ygUser* user = ygWorld::World->user(i);
		//if the user is within the space include in mirroring
		if ((user->head()) &&
		(contains(user->head()->origin(this))))
			{
			inside.insert(inside.begin(),user);
			if (prevInside.find(user) == prevInside.end())
				dcsUsers->addChild(user->pfnode());
			}
		}
	set<ygUser*>::const_iterator iter;
	//if user has left the space then remove them from mirroring
	for (iter = prevInside.begin(); iter != prevInside.end(); ++iter)
		{
		if (inside.find(*iter) == inside.end())
			dcsUsers->removeChild((*iter)->pfnode());
		}
	prevInside = inside;
	pfMatrix mirrorMatrix;
	getTransform(mirrorMatrix);
	pfMatrix inverseMirrorMatrix;
	inverseMirrorMatrix.invertOrtho(mirrorMatrix);
	pfMatrix matrix;
	matrix = inverseMirrorMatrix * reflection;
	dcsUsers->setMat(matrix);
	//for each subject
	for (int i=0;i<subjects.size();i++)
		{
		//get the subject matrix
		pfMatrix subjectMatrix;
		subjects[i]->getTransform(subjectMatrix);
		//get the subject inverse matrix
		pfMatrix inverseSubjectMatrix;
		inverseSubjectMatrix.invertOrtho(subjectMatrix);
		//construct the mirror matrix
		matrix = subjectMatrix * inverseMirrorMatrix * reflection;
		pfDCS* dcsSubject = (pfDCS*)pfnode()->getChild(i+1);
		//set the subject matrix
		dcsSubject->setMat(matrix);
		}
	ygSpace::app();
	}

void mirror::addSubject(const ygString& newSubject)
	{
	//create a new DCS for the subject
	ygNode* subjectNode = ygNodeDB::find(newSubject);
	pfDCS* dcsSubject = new pfDCS;
	//add an instance of the subject under the DCS
	dcsSubject->addChild(subjectNode->pfnode());
	//add the DCS as a child of this node
	pfnode()->addChild(dcsSubject);
	//add the subject to the list of subjects
	subjects.push_back(subjectNode);
	}

void mirror::acceptNetKey(const ygString& key)
	{
	if (key == "subject")
		{
                int i = 0;
                int currChild = 1;
		ygString* currSubject;
                while (currSubject = subjectList.nextToken(" ",&i))
                        {
                        if (currChild > subjects.size())
                                addSubject(*currSubject);
                        currChild++;
                        delete currSubject;
                        }
		}
	else if (key == "axis")
		{
		setAxis();
		}
	else
		ygSpace::acceptNetKey(key);
	}
