// Description: adjusts the material properties of all nodes below it
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the "relative" option makes changes relative to the current material property
//<li> the material property of all instanced object nodes will be changed
//<li> use the cache option on child objects to avoid unwanted side effects
//</ul>
//
// Category: Attributes
// Author: Alex Hill
//           11/01/01
// Revision: 04/01/04 Alex Hill - added "alpha" message to adjust transparency
//
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfGroup.h>
#include <Performer/pr/pfMaterial.h>
#include <Performer/pf/pfChannel.h>
#include "ygNetKeys.h"
#include "materialProperty.h"

extern "C" ygNode* construct_materialProperty(const char* name,bool master) { return new ygMaterialProperty(name,master); }

ygMaterialProperty::ygMaterialProperty(const char* name,bool master) : ygNode(name,master)
	{
	gSetList = new pfList;
	ambient.set(0,0,0,0);
	diffuse.set(0,0,0,0);
	emission.set(0,0,0,0);
	specular.set(0,0,0,0);
	alpha.set(0,0);
	shininess.set(0,0);
	setClassName("materialProperty",true);
	addNetKey("ambient",&ambient,YG_NET_VEC4);
	addNetKey("diffuse",&diffuse,YG_NET_VEC4);
	addNetKey("emission",&emission,YG_NET_VEC4);
	addNetKey("specular",&specular,YG_NET_VEC4);
	addNetKey("alpha",&alpha,YG_NET_VEC2);
	addNetKey("shininess",&shininess,YG_NET_VEC2);
	}

ygMaterialProperty::~ygMaterialProperty(void)
	{
	}

void ygMaterialProperty::reset(void)
	{
	//reset all colors and shininess to zero
	gSetList = new pfList;
	ambient.set(0,0,0,0);
	diffuse.set(0,0,0,0);
	emission.set(0,0,0,0);
	specular.set(0,0,0,0);
	alpha.set(0,0);
	shininess.set(0,0);
	} 

void ygMaterialProperty::message(const ygMessage& msg)
	{
	//set or change the ambient color of the material
	if (msg == "ambient")
		{
		if (msg.args.size() > 2)
			{
			ambient[0] = msg.floatArg(0);
			ambient[1] = msg.floatArg(1);
			ambient[2] = msg.floatArg(2);
			if (msg.args.size() > 3 && 
				(msg.args[3] == "rel" || msg.args[3] == "relative"))
				{
				changeColor(GL_AMBIENT,ambient[0],ambient[1],ambient[2]);
				ambient[3] = 1.0;
				}
			else
				{
				setColor(GL_AMBIENT,ambient[0],ambient[1],ambient[2]);
				ambient[3] = 0.0;
				}
			netKeyChanged("ambient");
			}
		}
	//set or change the diffuse color of the material
	else if (msg == "diffuse")
		{
		if (msg.args.size() > 2)
			{
			diffuse[0] = msg.floatArg(0);
			diffuse[1] = msg.floatArg(1);
			diffuse[2] = msg.floatArg(2);
			if (msg.args.size() > 3 && 
				(msg.args[3] == "rel" || msg.args[3] == "relative"))
				{
				changeColor(GL_DIFFUSE,diffuse[0],diffuse[1],diffuse[2]);
				diffuse[3] = 1.0;
				}
			else
				{
				setColor(GL_DIFFUSE,diffuse[0],diffuse[1],diffuse[2]);
				diffuse[3] = 0.0;
				}
			netKeyChanged("diffuse");
			}
		}
	//set or change the emission color of the material
	else if (msg == "emission")
		{
		if (msg.args.size() > 2)
			{
			emission[0] = msg.floatArg(0);
			emission[1] = msg.floatArg(1);
			emission[2] = msg.floatArg(2);
			if (msg.args.size() > 3 && 
				(msg.args[3] == "rel" || msg.args[3] == "relative"))
				{
				changeColor(GL_EMISSION,emission[0],emission[1],emission[2]);
				emission[3] = 1.0;
				}
			else
				{
				setColor(GL_EMISSION,emission[0],emission[1],emission[2]);
				emission[3] = 0.0;
				}
			netKeyChanged("emission");
			}
		}
	//set or change the specular color of the material
	else if (msg == "specular")
		{
		if (msg.args.size() > 2)
			{
			specular[0] = msg.floatArg(0);
			specular[1] = msg.floatArg(1);
			specular[2] = msg.floatArg(2);
			if (msg.args.size() > 3 && 
				(msg.args[3] == "rel" || msg.args[3] == "relative"))
				{
				changeColor(GL_SPECULAR,specular[0],specular[1],specular[2]);
				specular[3] = 1.0;
				}
			else
				{
				setColor(GL_SPECULAR,specular[0],specular[1],specular[2]);
				specular[3] = 0.0;
				}
			netKeyChanged("specular");
			}
		}
	//set or change the transparency of the material
	else if (msg == "alpha")
		{
		if (msg.args.size() > 0)
			{
			alpha[0] = msg.floatArg(0);
			if (msg.args.size() > 1 && 
				(msg.args[1] == "rel" || msg.args[1] == "relative"))
				{
				changeAlpha(alpha[0]);
				alpha[1] = 1.0;
				}
			else
				{
				setAlpha(alpha[0]);
				alpha[1] = 1.0;
				}
			netKeyChanged("alpha");
			}
		}
	//set or change the shininess of the material
	else if (msg == "shininess")
		{
		if (msg.args.size() > 0)
			{
			shininess[0] = msg.floatArg(0);
			if (msg.args.size() > 1 && 
				(msg.args[1] == "rel" || msg.args[1] == "relative"))
				{
				changeShininess(shininess[0]);
				shininess[1] = 1.0;
				}
			else
				{
				setShininess(shininess[0]);
				shininess[1] = 1.0;
				}
			netKeyChanged("shininess");
			}
		}
	else
		ygNode::message(msg);
	}
	
void ygMaterialProperty::setGSetList(pfNode *node)
	{
	int i;
	//if node type is a pfGeode
	if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode *)node;
		//for each geoset
		for (i=0; i < geode->getNumGSets(); i++)
			{
			//add geoset to the geoset list
			pfGeoSet* gSet = geode->getGSet(i);
			gSetList->add(gSet);
			//duplicate geometry state to prevent material property sharing
			pfGeoState* gState = gSet->getGState();
			gSetList->add(gSet);
			pfGeoState* newGState = new pfGeoState(*gState);
			gSet->setGState(newGState);
			gState = newGState;
			pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
			if (frontMtl)
				{
				pfMaterial* newMtl = new pfMaterial(*frontMtl);
				gState->setAttr(PFSTATE_FRONTMTL,newMtl);
				gState->setAttr(PFSTATE_BACKMTL,newMtl);
				frontMtl = newMtl;
				}
			pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
			if (backMtl && backMtl != frontMtl)
				{
				pfMaterial* newMtl = new pfMaterial(*backMtl);
				gState->setAttr(PFSTATE_BACKMTL,newMtl);
				}
			}
		}
	//else, if node type is a pfGroup
	else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		//call setGSetList recursively on all children
			for (i=0; i < group->getNumChildren(); i++)
			setGSetList(group->getChild(i));
		}
	}

void ygMaterialProperty::setColor(int color,float r,float g,float b)
	{
	for (int i=0;i<gSetList->getNum();i++)
		{
		pfGeoSet* gSet = (pfGeoSet*)gSetList->get(i);
		pfGeoState* gState = gSet->getGState();
		pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
		pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
		int frontMode,backMode;
		if (frontMtl)
			{
			frontMode = frontMtl->getColorMode(PFMTL_FRONT);
			if (frontMode == PFMTL_CMODE_AMBIENT_AND_DIFFUSE)
				if (color == PFMTL_AMBIENT)
					frontMode = PFMTL_CMODE_AMBIENT;
				else if (color == PFMTL_DIFFUSE)
					frontMode = PFMTL_CMODE_DIFFUSE;
			if (color == frontMode)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					{
					colors[i][0] = r;
					colors[i][1] = g;
					colors[i][2] = b;
					}
				}
			else
				frontMtl->setColor(color,r,g,b);
			}
		if (backMtl)
			{
			backMode = backMtl->getColorMode(PFMTL_BACK);
			if (backMode == PFMTL_CMODE_AMBIENT_AND_DIFFUSE)
				if (color == PFMTL_AMBIENT)
					backMode = PFMTL_CMODE_AMBIENT;
				else if (color == PFMTL_DIFFUSE)
					backMode = PFMTL_CMODE_DIFFUSE;
			if (backMode != frontMode && color == backMode)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					{
					colors[i][0] = r;
					colors[i][1] = g;
					colors[i][2] = b;
					}
				}
			else
				backMtl->setColor(color,r,g,b);
			}
		}
    }
	
void ygMaterialProperty::changeColor(int color,float r,float g,const float b)
	{
	for (int i=0;i<gSetList->getNum();i++)
		{
		pfGeoSet* gSet = (pfGeoSet*)gSetList->get(i);
		pfGeoState* gState = gSet->getGState();
		pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
		pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
		int frontMode,backMode;
		if (frontMtl)
			{
			frontMode = frontMtl->getColorMode(PFMTL_FRONT);
			if (frontMode == PFMTL_CMODE_AMBIENT_AND_DIFFUSE)
				if (color == PFMTL_AMBIENT)
					frontMode = PFMTL_CMODE_AMBIENT;
				else if (color == PFMTL_DIFFUSE)
					frontMode = PFMTL_CMODE_DIFFUSE;
			if (color == frontMode)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					{
					colors[i][0] += r;
					colors[i][1] += g;
					colors[i][2] += b;
					}
				}
			else
				{
				float currR,currG,currB;
				frontMtl->getColor(color,&currR,&currG,&currB);
				frontMtl->setColor(color,currR+r,currG+g,currB+b);
				}
			}
		if (backMtl)
			{
			backMode = backMtl->getColorMode(PFMTL_BACK);
			if (backMode == PFMTL_CMODE_AMBIENT_AND_DIFFUSE)
				if (color == PFMTL_AMBIENT)
					backMode = PFMTL_CMODE_AMBIENT;
				else if (color == PFMTL_DIFFUSE)
					backMode = PFMTL_CMODE_DIFFUSE;
			if (backMode != frontMode && color == backMode)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					{
					colors[i][0] += r;
					colors[i][1] += g;
					colors[i][2] += b;
					}
				}
			else
				{
				float currR,currG,currB;
				backMtl->getColor(color,&currR,&currG,&currB);
				backMtl->setColor(color,currR+r,currG+g,currB+b);
				}
			}
		}
    }

void ygMaterialProperty::setAlpha(float a)
	{
	for (int i=0;i<gSetList->getNum();i++)
		{
		pfGeoSet* gSet = (pfGeoSet*)gSetList->get(i);
		gSet->setDrawBin(PFSORT_TRANSP_BIN);
		pfGeoState* gState = gSet->getGState();
		gState->setMode(PFSTATE_TRANSPARENCY, PFTR_BLEND_ALPHA);
		pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
		pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
		int frontMode,backMode;
		if (frontMtl)
			{
			frontMode = frontMtl->getColorMode(PFMTL_FRONT);
			if (frontMode != PFMTL_CMODE_OFF)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					colors[i][3] = a;
				}
			else
				frontMtl->setAlpha(a);
			}
		if (backMtl)
			{
			backMode = backMtl->getColorMode(PFMTL_BACK);
			if (backMode != PFMTL_CMODE_OFF)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					colors[i][3] = a;
				}
			else
				backMtl->setAlpha(a);
			}
		}
    }
	
void ygMaterialProperty::changeAlpha(float a)
	{
	for (int i=0;i<gSetList->getNum();i++)
		{
		pfGeoSet* gSet = (pfGeoSet*)gSetList->get(i);
		gSet->setDrawBin(PFSORT_TRANSP_BIN);
		pfGeoState* gState = gSet->getGState();
		gState->setMode(PFSTATE_TRANSPARENCY, PFTR_BLEND_ALPHA);
		pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
		pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
		int frontMode,backMode;
		if (frontMtl)
			{
			frontMode = frontMtl->getColorMode(PFMTL_FRONT);
			if (frontMode != PFMTL_CMODE_OFF)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					colors[i][3] += a;
				}
			else
				{
				float currA;
				currA = frontMtl->getAlpha();
				frontMtl->setAlpha(currA+a);
				}
			}
		if (backMtl)
			{
			backMode = backMtl->getColorMode(PFMTL_BACK);
			if (backMode != PFMTL_CMODE_OFF)
				{
				ushort* indexList;
				pfVec4* colors;
				gSet->getAttrLists(PFGS_COLOR4,(void **)&colors,&indexList);
				int numColors = pfGetSize(colors)/sizeof(pfVec4);
				for (int i=0;i<numColors;i++)
					colors[i][3] += a;
				}
			else
				{
				float currA;
				currA = backMtl->getAlpha();
				backMtl->setAlpha(currA+a);
				}
			}
		}
    }
void ygMaterialProperty::setShininess(float s)
	{
	for (int i=0;i<gSetList->getNum();i++)
		{
		pfGeoSet* gSet = (pfGeoSet*)gSetList->get(i);
		pfGeoState* gState = gSet->getGState();
		pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
		pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
		if (frontMtl)
			frontMtl->setShininess(s);
		if (backMtl)
			backMtl->setShininess(s);
		}
	}
	
void ygMaterialProperty::changeShininess(float s)
	{
	for (int i=0;i<gSetList->getNum();i++)
		{
		pfGeoSet* gSet = (pfGeoSet*)gSetList->get(i);
		pfGeoState* gState = gSet->getGState();
		pfMaterial* frontMtl = (pfMaterial*)gState->getAttr(PFSTATE_FRONTMTL);
		pfMaterial* backMtl = (pfMaterial*)gState->getAttr(PFSTATE_BACKMTL);
		if (frontMtl)
			{
			float currS = frontMtl->getShininess();
			frontMtl->setShininess(currS+s);
			}
		if (backMtl)
			{
			float currS = backMtl->getShininess();
			backMtl->setShininess(currS+s);
			}
		}
	}
	
void ygMaterialProperty::app(void)
	{
	//if first frame then set the geoset list
	if (pfChanged())
		setGSetList(pfnode());
	}

void ygMaterialProperty::acceptNetKey(const ygString& key)
	{
	if (key == "ambient")
		{
		if (ambient[3] > 0.5)
			changeColor(GL_AMBIENT,ambient[0],ambient[1],ambient[2]);
		else
			setColor(GL_AMBIENT,ambient[0],ambient[1],ambient[2]);
		}
	else if (key == "diffuse")
		{
		if (diffuse[3] > 0.5)
			changeColor(GL_DIFFUSE,diffuse[0],diffuse[1],diffuse[2]);
		else
			setColor(GL_DIFFUSE,diffuse[0],diffuse[1],diffuse[2]);
		}
	else if (key == "emission")
		{
		if (emission[3] > 0.5)
			changeColor(GL_EMISSION,emission[0],emission[1],emission[2]);
		else
			setColor(GL_EMISSION,emission[0],emission[1],emission[2]);
		}
	else if (key == "specular")
		{
		if (specular[3] > 0.5)
			changeColor(GL_SPECULAR,specular[0],specular[1],specular[2]);
		else
			setColor(GL_SPECULAR,specular[0],specular[1],specular[2]);
		}
	else if (key == "shininess")
		{
		if (shininess[1] > 0.5)
			changeShininess(shininess[0]);
		else
			setShininess(shininess[0]);
		}
	else
		ygNode::acceptNetKey(key);
	}

