//Description: displays a live video texture to a rectangular surface, using the JPEG decompression
//
//<b>notes:</b>
//<ul>
//<li> this node is currently supported only on LINUX machines, using the xine library
//<li> uses the JPEGDecompress C++ class to decompress jpeg data
//<li> the image is received through network via a non-blocking UDP socket
//<li> the image is divided into several small packets
//<li> the data is compressed with jpeg
//<ul> Requirements:
//<li> A Windows PC
//<li> A Web Camera, connected to the Windows PC. In general any Web Camera should be able to work with the winAvatar Application, but the Web Cameras that have been tested are:
//<ul><li>Logitech QuickCam Pro 3000.
//    <li>Phillips ToUcam.
//</ul>
//<li> The <a href="http://www.evl.uic.edu/eleni/yg/winAvatarCam.html">WinAvatarCam Application</a> for capturing and streaming video.  
//<li> Check whether UDP data are actually received in the Linux machine, which actually runs the Ygdrasil environment. Two examples of user-friendly applications are:
//<ul><li><a href="http://web.wt.net/~billw/gkrellm/gkrellm.html">GKrellM </a>
//    <li><a href="http://sourceforge.net/projects/xosview/">xosview </a>
//</ul>
//</ul>
//
//
// Category: Attributes
// Author: Kalle Jalkanen and Jonas Westling, Interactive Institute Ume, Sweden
//           04/26/02
// Revision: 08/10/03 Helen-Nicole Kostis - documentation
//
//

#include <CAVERNnet_udp_c.hxx>
#include <CAVERN.hxx>
#include <errno.h>
#include "liveVideo.h"
#include "JPEGDecompress.h"
#include "rectSurface.h"

using namespace std;

extern "C" ygNode* construct_liveVideo(const char* name,bool master)
{ 
	return new liveVideo(name,master); 
}


struct _liveVideoPrivateData
{
	JPEGDecompress djpeg;		
	rectSurface *surface;
	bool on;          
	CAVERNnet_udp_c *camsocket;	
	int camListenPort;				
	CAVERNnet_udp_c *ygsocket;		
	int ygListenPort;					
	unsigned char packet[65536];	
	unsigned char cdata[65536];   
	unsigned char tempdata[65536]; 
	int tempbytes;	
	int nrOfPackets;
	int byteCounter;
	int client_sent;
	float startTime;
	float noiseTime;
	bool localOn;
	CAVERNnet_udp_c *vcsocket;
	ygString avatarID;
	float vcTime;
	ygString sizeString; 
	int first;

};




liveVideo::liveVideo(const char* name, bool master) : ygSpace(name, master)
{
	setClassName("liveVideo", true); 
   	_p = new struct _liveVideoPrivateData;
	ygString nodeName;
	createDummyName(nodeName);
	_p->surface = new rectSurface(nodeName.c_str(), false);
	_p->first = 1;
	_p->on = false;
	addNetKey("toggle", &_p->on, YG_NET_BOOL);	
	_p->camListenPort = -1;
	_p->camsocket = NULL;
	_p->ygListenPort = -1;
	addNetKey("ygserverport", &_p->ygListenPort, YG_NET_INT);
	_p->ygsocket = NULL;	
	if (_p->djpeg.init()>0)
		cerr << "djpeg.init()" << endl;
	_p->tempbytes = 0;	
	_p->nrOfPackets = -1;
	_p->byteCounter = 0;

	_p->client_sent = 0;
	_p->startTime = 0;
	_p->noiseTime = 0;
	_p->localOn = false;	
	_p->vcsocket = NULL;
	_p->avatarID = "OFF";
	_p->vcTime = 0;
	addNetKey("avatarid", &_p->avatarID, YG_NET_STRING);
	_p->sizeString = "";
	addNetKey("coords", &_p->sizeString, YG_NET_STRING);
}



liveVideo::~liveVideo()
{
	delete _p->surface;
	
	if (_p->vcsocket) 
	{
		delete _p->vcsocket;
	}
	
	if (_p->ygsocket) 
	{
		delete _p->ygsocket;
	}
	
	if (_p->camsocket) 
	{
		delete _p->camsocket;
	}

   delete _p;

}


void liveVideo::reset()
{
	_p->surface->reset();
	ygSpace::reset();
}




void liveVideo::message(const ygMessage& msg)
{
	if (msg == "toggle") 
	{	
		_p->on = !_p->on;
		if (!_p->on) 
		{
			_p->surface->clear(0L);
			_p->surface->setImg(NULL, 0, 0, 0);
		}
		netKeyChanged("toggle");
	} 
	else if (msg == "on") 
	{
		_p->on = true;
		netKeyChanged("toggle");	
	} 
	else if (msg == "off") 
	{
		_p->surface->clear(0L);
		_p->surface->setImg(NULL, 0, 0, 0);
		_p->on = false;
		netKeyChanged("toggle");
	} 
	else if (msg == "camlistenport") 
	{
		if (!_p->camsocket) 
		{
			_p->camListenPort = (int)msg.intArg(0);
			_p->camsocket = new CAVERNnet_udp_c(); 
			_p->camsocket->init(_p->camListenPort);
		}
	} 
	else if (msg == "yglistenport") 
	{
			_p->ygListenPort = (int)msg.intArg(0);
			_p->ygsocket = new CAVERNnet_udp_c(); 
			_p->ygsocket->init(_p->ygListenPort);	

			netKeyChanged("ygserverport");
			
	} 
	else if (msg == "avatarid")
	{ 
		if ( msg.args.size() != 1 )
		{
			msg.error("","(wrong number of arguments - should be 'avatarid(name)' )");
		} 
		else 
		{
			_p->avatarID = msg.stringArg(0);
			netKeyChanged("avatarid");
		}
		char *voicecomm_server = getenv("VOICECOMM_SERVER");
		char *voicecomm_port = getenv("VOICECOMM_PORT");
		if (voicecomm_server && voicecomm_port) 
		{
			ygString host(voicecomm_server);
			int port = atoi(voicecomm_port);
			_p->vcsocket = new CAVERNnet_udp_c();
			_p->vcsocket->init();
			_p->vcsocket->setSendAddress(host.c_str(), port);
		} 
		else 
		{
			cerr << "VOICECOMM_SERVER and/or VOICECOMM_PORT not defined" << endl;
		}

	} 
	else if (msg == "coords") 
	{
		_p->surface->message(msg);
		msg.writeString(_p->sizeString);
		netKeyChanged("coords");	
	} 
	else 
	{
		ygSpace::message(msg);
	}

}


#define MAXPAYLOAD 1400
#define HEADERSIZE 7


int liveVideo::readImageChunks()
{
	if (!_p->camsocket)
	{
		return 0;
	}

	int bytesRecv = _p->camsocket->receive((char *)_p->packet, 65535, CAVERNnet_udp_c::NON_BLOCKING);

	
	if (bytesRecv > 0) 
	{
		int imageNr = (int)_p->packet[0];   
		int picPartNr = (int)_p->packet[1];
		int isLast = (int)_p->packet[2];
		
		if (picPartNr == 0) 
		{
			if (_p->nrOfPackets != -1) 
			{
				cerr << "We lost the last packet" << endl;
			}
			_p->byteCounter = 0;
			_p->nrOfPackets = 0;	
		} 
		else 
		{
			_p->nrOfPackets++;
		}
		
		memcpy(_p->tempdata + picPartNr*MAXPAYLOAD, _p->packet + HEADERSIZE, bytesRecv - HEADERSIZE);	
		_p->byteCounter += (bytesRecv - HEADERSIZE);	
		

		if (isLast && (picPartNr==_p->nrOfPackets)) 
		{
			_p->nrOfPackets = -1;
			return _p->byteCounter;
			
		} 
		else 
		{
			if (isLast) 
			{		
				cerr<<imageNr<<" Lost part of a this picture: got (";
				cerr<<_p->nrOfPackets+1<<"/"<<picPartNr+1<<") packets"<<endl;
				_p->nrOfPackets = -1;					
			}
		}
	
	} 
	else if (bytesRecv != CAVERNnet_udp_c::NON_BLOCKING_HAS_NO_DATA) 
	{
		cerr << "Server: CAVERN error: UDP socket receive failed (code=" << bytesRecv << ")" << endl;	
		delete _p->camsocket;
		_p->camsocket = NULL;	
		_p->nrOfPackets = -1;
		_p->byteCounter = 0;
	}

	return 0;
}



void liveVideo::checkVolume()
{
	ygNode *viewer = ygWorld::World->viewer();

	if (viewer) 
	{
		ygString mess("OFF");	
		_p->localOn = false;
		pfVec3 userPos = viewer->origin(this);
		if (contains(userPos) && _p->on) 
		{
			_p->localOn = true;
			mess = _p->avatarID;	
		} 

		float Hz = 5.0;
		float t = (ygWorld::FrameTime - _p->vcTime) * Hz;

		if (t >= 1.0) 
		{
			if (_p->vcsocket)
				_p->vcsocket->send(mess.c_str(), mess.length()+1, CAVERNnet_udp_c::NON_BLOCKING);		
			_p->vcTime = ygWorld::FrameTime;
		}	


	}
}


void liveVideo::app(void)
{
	int numbytes = 0;

	checkVolume();
	
	if (_p->on) 
	{	
		if (isNetMaster()) 
		{
			numbytes = readImageChunks();

			if (numbytes > 0) 
			{
				memcpy(_p->cdata, _p->tempdata, numbytes);
				_p->tempbytes = numbytes;
			}

			if (_p->ygsocket) 
			{
				char hello_buf[128];
				int val = _p->ygsocket->receive((char *)hello_buf, 128, CAVERNnet_udp_c::NON_BLOCKING);
				if ((val>0) && _p->tempbytes) 
				{
					if (!strncmp("HELLO_MASTER", hello_buf, 12)) 
					{
						_p->ygsocket->copyReceiveAddressToSendAddress();
						
						int ret = _p->ygsocket->send((char*)_p->cdata, _p->tempbytes, CAVERNnet_udp_c::NON_BLOCKING);
						if (ret>0) 
						{
							
						}

					}
				}
			}
		} 
		else if (_p->localOn) 
		{
			char hello_msg[] = "HELLO_MASTER";

			if (_p->ygsocket) 
			{
				if (!_p->client_sent) 
				{
					int val = _p->ygsocket->send((char *)hello_msg, strlen(hello_msg), CAVERNnet_udp_c::NON_BLOCKING);
					if (val>0) 
					{
						_p->client_sent = 1;	
						_p->startTime = ygWorld::FrameTime;
					}
				}

				if (_p->client_sent) 
				{				
					numbytes = _p->ygsocket->receive((char *)_p->cdata, 65535, CAVERNnet_udp_c::NON_BLOCKING);
					if (numbytes > 0) 
					{

					} 
					else if (numbytes != CAVERNnet_udp_c::NON_BLOCKING_HAS_NO_DATA) 
					{
						cerr << "Client: CAVERN error: UDP socket receive failed." << endl;
						delete _p->ygsocket;
						_p->ygsocket = NULL;
					} 
					else 
					{
					}
				}

				
				float Hz = 10.0;
				float t = (ygWorld::FrameTime - _p->startTime) * Hz;

				if (t >= 1.0) {
					_p->client_sent = 0;
				}	
			}
		}
	}

	if ( _p->surface->isCreated() ) 
	{

		if (_p->first) 
		{
			pfnode()->addChild(_p->surface->pfnode());
			_p->first = 0;
		}
		
		if (_p->localOn) 
		{
		
			if (_p->on) 
			{
				if (numbytes > 0) 
				{
					_p->djpeg.decompress(_p->cdata, numbytes);
					int w = _p->djpeg.getWidth();
					int h = _p->djpeg.getHeight();

					_p->surface->setImg(_p->djpeg.data(), w, h, 3);
				}

			}
		
		}
	}

	_p->surface->app();
	ygSpace::app();
}



void liveVideo::acceptNetKey(const ygString& key)
{
	if (key == "toggle") 
	{
		if (!_p->on) 
		{
			_p->surface->clear(0L);
			_p->surface->setImg(NULL, 0, 0, 0);
		}
			
	} 
	else if (key == "ygserverport") 
	{

		if (_p->ygsocket == NULL) 
		{
			char *scene_host = getenv("SCENE_HOST");

			if (scene_host) 
			{
				ygString hostname(scene_host);
				_p->ygsocket = new CAVERNnet_udp_c;
				_p->ygsocket->init();

				_p->ygsocket->setSendAddress(hostname.c_str(), _p->ygListenPort);
			} 
			else 
			{
				cerr << "SCENE_HOST not defined" << endl; 
			}
		}
	} 
	else if (key == "avatarid") 
	{

		if ( _p->vcsocket == NULL ) 
		{

			char *voicecomm_server = getenv("VOICECOMM_SERVER");
			char *voicecomm_port = getenv("VOICECOMM_PORT");
			if (voicecomm_server && voicecomm_port) {
				ygString host(voicecomm_server);
				int port = atoi(voicecomm_port);
				_p->vcsocket = new CAVERNnet_udp_c();
				_p->vcsocket->init();
				_p->vcsocket->setSendAddress(host.c_str(), port);
			} 
			else 
			{
				cerr << "VOICECOMM_SERVER and/or VOICECOMM_PORT not defined" << endl;
			}
		}
	} 
	else if (key == "coords") 
	{
		ygMessage msg;
		msg.parseString(_p->sizeString);	
		_p->surface->message(msg);
	} 
	else 
	{
		ygSpace::acceptNetKey(key);		
	}
}







