// Description: a selector based on distance from the viewer
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the first range value defines when the first child is selected
//<li> all subsequent range values define the selection of the next child
//<li> the last range value defines when the last child is unselected
//<li> the total number of range values should be the number of children plus one
//</ul>
//
// Category: Selection
// Author: Alex Hill
// Revision: 11/01/01
//
#include <ygWorld.h>
#include "levelOfDetail.h"

extern "C" ygNode* construct_levelOfDetail(const char* name,bool master) { return new levelOfDetail(name,master); }

levelOfDetail::levelOfDetail(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("levelOfDetail",true);
	sw = new pfSwitch;
	setPfNode(sw);
	debugFlag("selected",&debugSelected);
	}

levelOfDetail::~levelOfDetail(void)
	{
	}


void levelOfDetail::reset(void)
	{
	ranges.clear();
	//set the range to the first child as 0.0
	ranges.push_back(0.0);
	//initialize a range variable to 50.0
	float range = 50.0;
	//for each child
	for (int i=0;i<numChildren();i++)
		{
		//set child number plus one range to the variable
		ranges.push_back(range);
		//double the range variable
		range *= 2.0;
		}
	selected = PFSWITCH_OFF;
	sw->setVal(selected);
	prevSelected = selected;
	ygNode::reset();
	}


void levelOfDetail::message(const ygMessage& msg)
	{
	//set the distance at which a child is swed on
	if (msg == "range")
		{
		if (msg.args.size() > 1) 
			{
			float newRange = msg.floatArg(0);
			int child = msg.intArg(1);
			if (child >= 0 && child <= numChildren())
				ranges[child] = newRange;
			else
				msg.error(name(),"(invalid child node)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the distance for each child plus sw off distance
	else if (msg == "ranges")
		{
		if (msg.args.size() > 0)
			{
			int children = msg.args.size();
			if (children > numChildren())
				children = numChildren();
			for (int i=0;i<=children;i++) 
				ranges[i] = msg.floatArg(i);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}
	
void levelOfDetail::app(void)
	{
	//get the world viewer
	ygNode* viewer = ygWorld::World->viewer();
	//find the viewer with respect to this node
	pfVec3 position = origin(viewer);
	//find the distance to the viewer
	float distance = position.length();
	//set selected to OFF
	selected = PFSWITCH_OFF;
	//for each range value minus the last one
	for (int i=0;i<ranges.size()-1;i++)
		{
		//if distance greater than range, then set selected to that child
		if (distance >= ranges[i])
			selected = i;
		}
	//if distance greater than last range, then set selected to OFF
	if (distance >= ranges[ranges.size()-1])
		selected = PFSWITCH_OFF;
	//if selected has changed
	if (selected != prevSelected)
		{
		//set sw value
		sw->setVal(selected);
		if (debugSelected)
			cout << "levelOfDetail " << name() << ": new value is "
				<< selected << endl;
		prevSelected = selected;
		}
	ygNode::app();
	}

void levelOfDetail::appTraverse(void)
	{
	//start timing
	timingBegin();
	//execute app method
	app();
	ygNode *sel = child(selected);
	//call appTraverse recursively on selected child
	if (sel)
		sel->appTraverse();
	//end timing
	timingEnd();
	}
