// Description: an array that interpolates between a start array and an end array
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> all input values generate linearly interpolated values in the value array
//<ul>
//<li> input values of 0.0 correspond to the start values
//<li> input values of 1.0 correspond to the end values
//</ul>
//<li> the input values can either be given by a message or taken from a child value node
//<li> see the <a href="value.html">value</a> node for more information
//</ul>
//
// Category: Math
// Author: Alex Hill
// Revision: 11/01/01
//
#include <ygWorld.h>
#include "interpolate.h"

extern "C" ygNode* construct_interpolate(const char* name,bool master) { return new interpolate(name,master); }

interpolate::interpolate(const char* name,bool master) : value(name,master)
	{
	setClassName("interpolate");
	startValues = new float;
	endValues = new float;
	startValues[0] = 0.0;
	endValues[0] = 1.0;
	}

interpolate::~interpolate(void)
	{
	}

void interpolate::reset(void)
	{
	//set start and end size to 1
	setSize(startValues,1,size);
	setSize(endValues,1,size);
	//initialize start value to 0.0
	startValues[0] = 0.0;
	//initialize end value to 1.0
	endValues[0] = 1.0;
	value::reset();
	}
	
void interpolate::setValueSize(int newSize)
	{
	setSize(values,newSize,size);
	setSize(lastValues,newSize,size);
	setSize(startValues,newSize,size);
	setSize(endValues,newSize,size,1.0);
	size = newSize;
	}	

void interpolate::message(const ygMessage& msg)
	{
	//set the input for all or a single array element
	if (msg == "input")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0 && element < size)
					setInterpolate(newValue,element);
				else
					msg.error(name(),"(index out of range)");
				}
			else
				{
				for (int i=0;i<size;i++)
					setInterpolate(newValue,i);
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the input for all array elements
	else if (msg == "inputs")
		{
		if (msg.args.size() > 0)
			{
			int numInputs = msg.args.size();
			if (numInputs == size)
				{
				for (int i=0;i<size;i++)
					{
					float newValue = msg.floatArg(i);
					setInterpolate(newValue,i);
					}
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the value of a single start array element
	else if (msg == "startValue")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0)
					{
					if (element >= size)
						setValueSize(element+1);
					startValues[element] = newValue;
					}
				else
					msg.error(name(),"(index out of range)");
				}
			else
				startValues[0] = newValue;
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the values of all start array elements
	else if (msg == "startValues")
		{
		if (msg.args.size() > 0)
			{
			int newSize = msg.args.size();
			if (newSize >= 0)
				{
				setValueSize(newSize);
				for (int i=0;i<size;i++) 
					startValues[i] = msg.floatArg(i);
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the value of a single end element
	else if (msg == "endValue")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0)
					{
					if (element >= size)
						setValueSize(element+1);
					endValues[element] = newValue;
					}
				else
					msg.error(name(),"(index out of range)");
				}
			else
				endValues[0] = newValue;
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the values of all end array elements
	else if (msg == "endValues")
		{
		if (msg.args.size() > 0)
			{
			int newSize = msg.args.size();
			if (newSize >= 0)
				{
				setValueSize(newSize);
				for (int i=0;i<size;i++) 
					endValues[i] = msg.floatArg(i);
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		value::message(msg);
	}
	
void interpolate::app(void)
	{
	//if this node has size greater than zero and has children
	if (size > 0 && numChildren() > 0 && child(numChildren()-1) && child(numChildren()-1)->isOfClass("value"))
		{
		value* currChild = (value*)child(numChildren()-1);
		int currSize = currChild->getSize();
		//if child size is greater than one
		if (currSize > 1)
			{
			//for each value in the last child node do
			for (int i=0;i<currSize;i++)
				{
				//if the element is less than the size of this node then
				if (i < size)
					{
					//set the element value to the interpolated value of the child value
					setInterpolate(currChild->getValue(i),i);
					}
				}
			}
		else
			{
			//retrieve the value of the child node
			float inputValue = currChild->getValue();
			//for each element in the array do
			for (int i=0;i<size;i++)
				{
				//set the element value to the interpolated value of the value
				setInterpolate(inputValue,i);
				}
			}
			
		}
	value::app();
	}
