// Description: a transform that moves down to the first floor intersection found
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can cause unpredictable behavior if the nodes being moved to the floor are also set as floor
//<li> the object can be oriented to the slope of the floor by setting the upright mode to false
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/01
//
#include <Performer/pf/pfScene.h>
#include "ygWorld.h"
#include "ygUtil.h"
#include "gravity.h"

extern "C" ygNode* construct_gravity(const char* name,bool master) { return new gravity(name,master); }


gravity::gravity(const char* name,bool master) : ygTransform(name,master)
	{
	setClassName("gravity");
	height = 0.0;
	speed = 1.0;
	fly = false;
	upright = true;
	}


void gravity::reset(void)
	{
	//set height to zero 
	height = 0.0;
	//set speed to 1.0 feet/second
	speed = 1.0;
	fly = false;
	//set upright to true
	upright = false;
	ygTransform::reset();
	}


void gravity::message(const ygMessage& msg)
	{
	//set the angle of the object to the surface or upright
	if (msg == "upright")
		{
		if (msg.args.size() > 0)
			setUpright(msg.boolArg(0));
		else
			setUpright(true);
		}
	//toggle the upright mode
	else if (msg == "toggleUpright")
			setUpright(!upright);
	//set the fly mode to enable or disable gravity
	else if (msg == "fly")
		{
		if (msg.args.size() > 0)
			setFly(msg.boolArg(0));
		else
			setFly(true);
		}
	//toggle the fly mode
	else if (msg == "togglefly")
			setFly(!fly);
	//set the speed in feet/second
	else if (msg == "speed")
		setSpeed(msg.floatArg(0));
	//set the height above the surface
	else if (msg == "height")
		setHeight(msg.floatArg(0));
	else
		ygTransform::message(msg);
	}

void gravity::setFly(const bool& value)
	{
	fly = value;
	}

void gravity::setUpright(const bool& value)
	{
	upright = value;
	}

void gravity::setHeight(const float& value)
	{
	height = value;
	}

void gravity::setSpeed(const float& value)
	{
	speed = value;
	}

void gravity::app(void)
	{
	//if not in fly mode then check for ground
	if (!fly)
		checkGround();
	ygTransform::app();
	}

void gravity::checkGround(void)
	{
	//set up intersection segment
	pfSegSet segset;
	pfHit **hits[32];
	segset.activeMask = 1;
	segset.isectMask = YG_ISECT_FLOORMASK;
	segset.discFunc = NULL;
	segset.bound = NULL;
	segset.mode = PFTRAV_IS_PRIM;
	if (!upright)
		segset.mode |= PFTRAV_IS_NORM;
	segset.segs[0].dir.set(0.0f, 0.0f, -1.0f);
	segset.segs[0].length = 5000.0f;

	//get self matix
	pfMatrix selfMatrix;
	getMatrix(selfMatrix);
	pfVec3 selfPos(0,0,0);
	selfPos.xformPt(selfPos, selfMatrix);

	//get parent matrix
	pfMatrix parentMatrix;
	getParentTransform(parentMatrix);
	pfVec3 parentPos(0,0,0);
	parentPos.xformPt(parentPos, parentMatrix);

	segset.segs[0].pos = parentPos + selfPos;
	//if an intersection with an object is found
	if (ygWorld::World->pfscene()->isect(&segset, hits))
		{
		// Get the intersection transformation matrix.
		pfVec3 isectPos,isectNorm;
		pfMatrix isectMatrix;
		(*hits[0])->query(PFQHIT_POINT, isectPos.vec);
		(*hits[0])->query(PFQHIT_XFORM, (float*)isectMatrix.mat);
		isectPos.xformPt(isectPos, isectMatrix);

		if (selfPos[PF_Z] > isectPos[PF_Z] - parentPos[PF_Z] + height + 2.0)
			selfPos[PF_Z] -= speed;
		else
			selfPos[PF_Z] = isectPos[PF_Z] - parentPos[PF_Z] + height;
		selfMatrix.makeIdent();
		selfMatrix.setRow(3,0.0,0.0,selfPos[2],1.0);

		//if not upright then adjust the angle to the surface normal
		if (!upright)
			{
			(*hits[0])->query(PFQHIT_NORM, isectNorm.vec);
			isectMatrix.setRow(3,0.0,0.0,0.0,1.0);
			isectNorm.xformPt(isectNorm, isectMatrix);
			isectNorm.normalize();
	
			pfVec3 frontVector(0,1,0);
			pfVec3 headingVector;
			headingVector.xformPt(frontVector, parentMatrix);
			headingVector[2] = 0.0;
			headingVector.normalize();
			pfMatrix headingMatrix;
			headingMatrix.makeVecRotVec(headingVector,frontVector);
			isectNorm.xformPt(isectNorm,headingMatrix);
	
			pfVec3 upVector(0,0,1);
			pfMatrix orientationMatrix;
			orientationMatrix.makeVecRotVec(upVector,isectNorm); 
			selfMatrix = orientationMatrix * selfMatrix;
			}

		// Set the new transformation and return.
		setMatrix(selfMatrix);
		}
	}
