// Description: a transform that moves with respect to a wand or other transformed node
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node is useful for grabbing objects and moving them when used in conjunction with a <a href="wandTrigger.html">wandTrigger</a> node
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/01
//
#include <ygNodeDB.h>
#include "grabber.h"

using namespace std;

extern "C" ygNode *construct_grabber(const char *name, bool master) { return new grabber(name, master);}

grabber::grabber(const char *name, bool master) : ygSimpleTransform(name, master)
	{
	setClassName("grabber");
	matrix.makeIdent();
	node = NULL;
	}

grabber::~grabber(void)
	{
	}

void grabber::reset(void)
	{
	//reset matrix to identity
	matrix.makeIdent();
	///reset node pointer
	node = NULL;
	ygSimpleTransform::reset();
	}


void grabber::message(const ygMessage& msg)
	{
	//toggle the grab/drop state with respect to a node
	if (msg == "toggle")
		{
		if (msg.args.size() == 1)
			{
			if (nodeName == msg.args[0] && node)
				{
				drop();
				}
			else
				{
				nodeName = msg.args[0];
				node = ygNodeDB::find(nodeName);
				if (node)
					grab();
				else
					msg.error(name(),"(node not found)");
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
    	}
	//grab a node and move with repect to that node
	else if (msg == "grab")
		{
		if (msg.args.size() == 1)
			{
			nodeName = msg.args[0];
			node = ygNodeDB::find(nodeName);
			if (node)
				grab();
			else
				msg.error(name(),"(node not found)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
    	}
	//drop connection with a node or any node grabbed
	else if (msg == "drop")
		{
		if (msg.args.size() == 1)
			{
			ygString dropName = msg.args[0];
			if (dropName == nodeName)
				drop();
			}
		else
			drop();
		}
	else
		ygSimpleTransform::message(msg);
	}

void grabber::grab(void)
	{
	//if the node has been found
	if (node)
		{
		/*Now we calculate the relative matrix between the node and this object
		  We use the following formula to calculate our matrix...
		                     -1
		  matrix = M      * M
		            self     node
		*/
		pfMatrix node_matrix;
		node->getTransform(node_matrix);
		
		pfMatrix inverse_node_matrix;
		inverse_node_matrix.invertOrtho(node_matrix);

		pfMatrix self_matrix;
		getTransform(self_matrix);
		
		// Set this instance as "grabbed"
		matrix = self_matrix * inverse_node_matrix;
		ygString args("node=");
		args += node->name();
		eventOccurred("grab",args);
		}
	}

void grabber::drop(void)
	{
	//if a node is being tracked then drop
	if (node)
		{
		ygString args("node=");
		args += node->name();
		eventOccurred("drop",args);
		node = NULL;
		}
	}

void grabber::app(void)
	{
	if (node)
		{
		/*If we're still grabbed, move us and our node with the node
		                         -1
		   M = matrix * M     * M
		                 node    parent
		*/

		// Get the node's transformation matrix
		pfMatrix node_matrix;
		node->getTransform(node_matrix);

		// Get our object's transformation matrix relative to the world
		pfMatrix parent_matrix;
		getParentTransform(parent_matrix);
		parent_matrix.invertOrtho(parent_matrix);

		// Set the new transformation and return
		pfMatrix self_matrix = matrix * node_matrix * parent_matrix;
		setMatrix(self_matrix);
		}
	ygSimpleTransform::app();
	}
