// Description: a navigator that can grab and move with respect to another node
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> this node has three modes of operation
//<ul>
//<li> grab - the user is navigated with repect to another node
//<li> move - the user is navigated to another node at maximum speed
//<li> teleport - the user is navigated to another node immediately
//</ul>
//</ul>
//
// Category: User
// Author: Alex Hill
//           11/01/01
// Revision: 01/15/03 Alex Hill - added report message
//
#include <ygWorld.h>
#include <ygNodeDB.h>
#include <iostream.h>
#include "grabNavigator.h"

extern "C" ygNode* construct_grabNavigator(const char* name,bool master) { return new grabNavigator(name,master); }

grabNavigator::grabNavigator(const char* name,bool master) : ygCAVENavigator(name,master)
	{
	setClassName("grabNavigator");
	matrix.makeIdent();
	node = NULL;
	mode = GRAB;
	}

grabNavigator::~grabNavigator(void)
	{
	}

void grabNavigator::reset(void)
	{
	matrix.makeIdent();
	//reset node pointer
	node = NULL;
	//set mode to grab
	mode = GRAB;
	//set position distance threshold to 0.1
	posDistanceThresh = 0.1;
	//set position motion threshold to 0.5
	posMotionThresh = 0.5;
	//set heading distance threshold to 0.5
	headDistanceThresh = 0.5;
	//set heading motion threshold to 5.0
	headMotionThresh = 5.0;
	arrived = false;
	ygCAVENavigator::reset();
	}

void grabNavigator::message(const ygMessage& msg)
	{
	//toggle the grab/drop state with respect to a node
	if (msg == "toggle")
		{
		if (msg.args.size() == 1)
			{
			if (nodeName == msg.args[0] && node)
				drop();
			else
				{
				nodeName = msg.args[0];
				node = ygNodeDB::find(nodeName);
				if (node)
					grab();
				else
					msg.error(name(),"(node not found)");
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//grab a node and move with repect to that node
	else if (msg == "grab")
		{
		if (msg.args.size() == 1)
			{
			nodeName = msg.args[0];
			node = ygNodeDB::find(nodeName);
			if (node)
				grab();
			else
				msg.error(name(),"(node not found)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//drop connection with a node or any node grabbed
	else if (msg == "drop")
		{
		if (msg.args.size() > 0)
			{
			ygString dropName = msg.args[0];
			if (dropName == nodeName)
				drop();
			}
		else
			drop();
		}
	//switch modes or reset to grab mode
	else if (msg == "mode")
		{
		if (msg.args.size() > 0)
			{
			if (msg.args[0] == "grab")
				{
				mode = GRAB;
				if (node)
					grab();
				}
			else if (msg.args[0] == "teleport")
				mode = TELEPORT;
			else if (msg.args[0] == "move")
				mode = MOVE;
			else
				mode = GRAB;
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//report navigated position
	else if (msg == "report")
		{
		if (msg.args.size() == 0)
			{
			char val[32];
			pfVec3 pos;
			pos = data().position();
			ygString args("xpos=");
			sprintf(val,"%f",pos[0]);
			args += val;
			args += " ypos=";
			sprintf(val,"%f",pos[1]);
			args += val;
			args += " zpos=";
			sprintf(val,"%f",pos[2]);
			args += val;
			float heading;
			heading = data().heading();
			args += " heading=";
			sprintf(val,"%f",heading);
			args += val;
			eventOccurred("report",args);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygCAVENavigator::message(msg);
	}

void grabNavigator::grab(void)
	{
	//if the node has been found
	if (node)
		{
		/*Now we calculate the relative matrix between the node and this object.
		  We use the following formula to calculate our matrix...
							   -1
		  matrix = M	  * M
					  self	 node
		*/
		pfMatrix nodeMatrix;
		node->getTransform(nodeMatrix);
		
		pfMatrix inverseNodeMatrix;
		inverseNodeMatrix.invertOrtho(nodeMatrix);

		pfMatrix selfMatrix;
		getTransform(selfMatrix);
		
		//calculate the relative matrix
		matrix = selfMatrix*inverseNodeMatrix;
		ygString args("node=");
		args += node->name();
		eventOccurred("grab",args);
		}
	}
	
void grabNavigator::drop(void)
	{
	//if a node is being tracked then drop
	if (node)
		{
		ygString args("node=");
		args += node->name();
		eventOccurred("drop",args);
		node = NULL;
		}
	}

void grabNavigator::app(void)
	{
	if (node)
		{
		bool arrived = false;
		//if mode is teleport
		if (mode == TELEPORT)
			{
			//get node matrix
			pfMatrix nodeMatrix;
			node->getTransform(nodeMatrix);
			//get node position
			pfVec3 newPos(0,0,0);
			newPos.xformPt(newPos,nodeMatrix);
			//get node heading
			pfVec3 newDir(0,1,0);
			newDir.xformVec(newDir,nodeMatrix);
			float newHeading;
			newHeading = pfArcTan2(-newDir[0],newDir[1]);
			//set new position and heading
			data().setPosition(newPos);
			data().setHeading(newHeading);
			arrived = true;
			}
		else if (mode == MOVE)
			{
			//get node matrix
			pfMatrix nodeMatrix;
			node->getTransform(nodeMatrix);
			//get node position
			pfVec3 nodePos(0,0,0);
			nodePos.xformPt(nodePos,nodeMatrix);
			//get node heading
			pfVec3 nodeDir(0,1,0);
			nodeDir.xformVec(nodeDir,nodeMatrix);
			float nodeHeading;
			nodeHeading = pfArcTan2(-nodeDir[0],nodeDir[1]);
			//get delta position
			pfVec3 newPos;
			newPos = data().position();
			pfVec3 deltaPos;
			deltaPos = nodePos - newPos;
			float distance = deltaPos.length();
			//calculate the new move based on navigation speed
			if (distance > posDistanceThresh)
				{
				if (distance < posMotionThresh)
					deltaPos.scale(posDistanceThresh/distance,deltaPos);
				newPos += deltaPos*speed()*ygWorld::FrameDeltaTime;
				}
			else 
				newPos = nodePos;
			float newHeading;
			newHeading = data().heading();
			float deltaHeading;
			deltaHeading = nodeHeading - newHeading;
			if (deltaHeading > 180.0)
				deltaHeading -= 360.0;
			if (deltaHeading < -180.0)
				deltaHeading += 360.0;
			//calculate the new heading based on rotation speed
			if (fabsf(deltaHeading) > headDistanceThresh)
				{
				if (fabsf(deltaHeading) < headMotionThresh)
					{
					if (deltaHeading < 0)
						deltaHeading = -headDistanceThresh;
					else
						deltaHeading = headDistanceThresh;
					}
				newHeading += 0.25*deltaHeading*rotSpeed()*ygWorld::FrameDeltaTime;
				}
			else 
				{
				newHeading = nodeHeading;
				if (newPos == nodePos)
					arrived = true;
				}
			//set new position and heading
			data().setPosition(newPos);
			data().setHeading(newHeading);
			}
		else 
			{
			//get node matrix
			pfMatrix nodeMatrix;
			node->getTransform(nodeMatrix);
			//get new navigation matrix
			pfMatrix newMatrix;
			newMatrix = matrix*nodeMatrix;
			//get new position
			pfVec3 newPos(0,0,0);
			newPos.xformPt(newPos,newMatrix);
			//get new heading
			pfVec3 newDir(0,1,0);
			newDir.xformVec(newDir,newMatrix);
			float newHeading;
			newHeading = pfArcTan2(-newDir[0],newDir[1]);
			//set new position and heading
			data().setPosition(newPos);
			data().setHeading(newHeading);
			}
		if (arrived)
			{	
			ygString args("node=");
			args += node->name();
			eventOccurred("arrived",args);
			arrived = false;
			}
		ygNavigator::app();
		}
	else
		ygCAVENavigator::app();
	}
