// Description: a base class for geometry nodes
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> the wall mode is used to turn collision detection on
//<li> the floor mode is used to specify whether the model is treated as terrain
//<li> the draw mode is set to false, geometry is not drawn but collision detection can still be enabled
//</ul>
//
// Category: Geometry
// Author: Alex Hill
// Revision: 11/01/02
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfGeode.h>
#include <ygWorld.h>
#include <ygNetKeys.h>
#include <ygPFObjectCache.h>
#include <Performer/pfdu.h>
#include "geometry.h"

extern "C" ygNode* construct_geometry(const char* name,bool master) { return new geometry(name,master); }

geometry::geometry(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("geometry",true);
	isectMask = YG_ISECT_ALLMASK;
	drawn = true;
	wire = false;
	cull = PFCF_BACK;
	pfnode()->setTravMask(PFTRAV_ISECT,isectMask,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
	addNetKey("draw",&drawn,YG_NET_BOOL);
	addNetKey("isect",&isectMask,YG_NET_INT);
	addNetKey("wire",&wire,YG_NET_BOOL);
	addNetKey("cull",&cull,YG_NET_INT);
	}

geometry::~geometry(void)
	{
	}


void geometry::reset(void)
	{
	//set the remote drawn flag to true
	setDraw(true);
	//set the local draw traversal mask
	pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
	//set the wireframe flag to false
	setWireframe(false);
	//set the cull mode to back faces
	setCull(PFCF_BACK);
	//reset intersection mask to all
	isectMask = YG_ISECT_ALLMASK;
	//set the isect traversal mask
	pfnode()->setTravMask(PFTRAV_ISECT,isectMask,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	ygNode::reset();
	}


void geometry::message(const ygMessage& msg)
	{
	//set collision detection status
	if (msg == "wall")
		{
		if (msg.args.size() > 0)
			setWall(msg.boolArg(0));
		else
			setWall();
		}
	//set walk navigation status
	else if (msg == "floor")
		{
		if (msg.args.size() > 0)
			setFloor(msg.boolArg(0));
		else
			setFloor();
		}
	//set wireframe mode
	else if (msg == "wireframe")
		{
		if (msg.args.size() > 0)
			setWireframe(msg.boolArg(0));
		else
			setWireframe();
		}
	//set draw mode
	else if (msg == "draw")
		{
		if (msg.args.size() > 0)
			{
			if (msg.args[0] == "false" || msg.args[0] == "off")
				{
				pfnode()->setTravMask(PFTRAV_DRAW,0,PFTRAV_SELF,PF_SET);
				setDraw(false);
				}
			else if (msg.args[0] == "local")
				{
				pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
				setDraw(false);
				}
			else if (msg.args[0] == "remote")
				{
				pfnode()->setTravMask(PFTRAV_DRAW,0,PFTRAV_SELF,PF_SET);
				setDraw(true);
				}
			else if (msg.args[0] == "true" || msg.args[0] == "on")
				{
				pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
				setDraw(true);
				}
			}
		else
			{
			pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
			setDraw(true);
			}
		}
	//set cull mode
	else if (msg == "cull")
		{
		if (msg.args.size() > 0)
			{
			int mode = PFCF_BACK;
			if (msg.args[0] == "off" || msg.args[0] == "none")
				mode = PFCF_OFF;
			else if (msg.args[0] == "front")
				mode = PFCF_FRONT;
			else if (msg.args[0] == "back")
				mode = PFCF_BACK;
			else if (msg.args[0] == "both")
				mode = PFCF_BOTH;
			setCull(mode);
			}
		else
			setCull(PFCF_BACK);
		}
	else
		ygNode::message(msg);
	}

void geometry::app(void)
	{
	if (pfChanged())
		{
		setCull(cull);
		setWireframe(wire);
		}
	ygNode::app();
	}

void geometry::setGeoState(uint64_t mode, int state, pfNode* node)
	{
	int i;
	//if node type is pfGeode then set geoset properties
	if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode *)node;
		//for each geode, call setGeosetTexture
		for (i=0; i < geode->getNumGSets(); i++)
			{
			pfGeoSet* gset = geode->getGSet(i);
			gset->getGState()->setMode(mode,state);
			}
		}
	//else, if node type is pfGroup
	else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		//call this function recursively on all children
		for (i=0; i < group->getNumChildren(); i++)
			setGeoState(mode,state,group->getChild(i));
		}
	}

void geometry::setWall(bool val)
	{
	if (val)
		isectMask |= YG_ISECT_WALLMASK;
	else
		isectMask &= ~YG_ISECT_WALLMASK;
	pfnode()->setTravMask(PFTRAV_ISECT,isectMask,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	netKeyChanged("isect");
	}


void geometry::setFloor(bool val)
	{
	if (val)
		isectMask |= YG_ISECT_FLOORMASK;
	else
		isectMask &= ~YG_ISECT_FLOORMASK;
	pfnode()->setTravMask(PFTRAV_ISECT,isectMask,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	netKeyChanged("isect");
	}

void geometry::setDraw(bool val)
	{
	drawn = val;
	if (!isNetMaster())
		{
		if (drawn)
			pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
		else
			pfnode()->setTravMask(PFTRAV_DRAW,0,PFTRAV_SELF,PF_SET);
		}
	netKeyChanged("draw");
	}

void geometry::setWireframe(bool val)
	{
	wire = val;
	if (wire)
		setGeoState(PFSTATE_ENWIREFRAME,PF_ON,pfnode());
	else
		setGeoState(PFSTATE_ENWIREFRAME,PF_OFF,pfnode());
	netKeyChanged("wire");
	}

void geometry::setCull(int mode)
	{
	cull = mode;
	setGeoState(PFSTATE_CULLFACE,cull,pfnode());
	netKeyChanged("cull");
	}

void geometry::acceptNetKey(const ygString& key)
	{
	if (key == "draw")
		setDraw(drawn);
	else if (key == "isect")
		pfnode()->setTravMask(PFTRAV_ISECT,isectMask,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	else if (key == "wire")
		setWireframe(wire);
	else if (key == "cull")
		setCull(cull);
	else
		ygNode::acceptNetKey(key);
	}
