// Description: initializes a DCS from a named node within an OpenFlight file.
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> a source and node are specified
//<li> default source is called "fltSource"
//<li> default node is the name of the fltTransform
//<li> desired node transformation is given to fltTransform node
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/01
//
#include <Performer/pf/pfDCS.h>
#include "ygNetKeys.h"
#include "ygNodeDB.h"
#include "fltTransform.h"

extern "C" ygNode* construct_fltTransform(const char* name,bool master) { return new fltTransform(name,master); }

struct _fltTransformPrivateData
	{
	pfDCS * dcs;
	pfMatrix netMatrix;
	ygString sourcename;
	};


fltTransform::fltTransform(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("fltTransform",true);
	p_ = new struct _fltTransformPrivateData;
	p_->dcs = new pfDCS;
	setPfNode(p_->dcs);
	p_->netMatrix.makeIdent();
	addNetKey("matrix", &p_->netMatrix, YG_NET_MATRIX);
	addNetKey("source",&p_->sourcename,YG_NET_STRING);
	}


fltTransform::~fltTransform(void)
	{
	delete p_;
	}


void fltTransform::reset(void)
	{
	if (isNetMaster())
		{
		pfMatrix mat;
		mat.makeIdent();
		setMatrix(mat);
		}
	ygNode::reset();
	}


/*****  DCS functions *****/

void fltTransform::getMatrix(pfMatrix& mat) const
	{
	p_->dcs->getMat(mat);
	}


void fltTransform::setMatrix(pfMatrix& mat)
	{
	p_->dcs->setMat(mat);
	storeMatrixKey();
	}


void fltTransform::setPosition(float x,float y,float z)
	{
	p_->dcs->setTrans(x,y,z);
	storeMatrixKey();
	}


void fltTransform::setOrientation(float x,float y,float z)
	{
	p_->dcs->setRot(z,x,y);
	storeMatrixKey();
	}


void fltTransform::setSize(float s)
	{
	p_->dcs->setScale(s);
	storeMatrixKey();
	}


void fltTransform::setSize(float x,float y,float z)
	{
	p_->dcs->setScale(x,y,z);
	storeMatrixKey();
	}


/***** Message functions *****/

void fltTransform::message(const ygMessage& msg)
	{
	if (msg == "node")
	        {
	        ygString fltSource("fltSource");  
	        if (msg.args.size() == 0) 
	                extractTransform(fltSource,name(),0);
	        else if (msg.args.size() == 1)
	                extractTransform(fltSource,msg.args[0],1);
                else if (msg.args.size() == 2)
                        if (msg.args[1] == "child")
                           extractTransform(fltSource,msg.args[0],2);
                        else
                           extractTransform(fltSource,msg.args[0],0);
	        else
	                msg.error(name(),"(wrong number of arguments)");
	        }
	else if (msg == "source")
	        {
	        if (msg.args.size() == 1) 
	                extractTransform(msg.args[0],name(),0);
	        else if (msg.args.size() == 2)
	                extractTransform(msg.args[0],msg.args[1],1);
                else if (msg.args.size() == 3)
                        if (msg.args[2] == "child")
                           extractTransform(msg.args[0],msg.args[1],2);
                        else
                           extractTransform(msg.args[0],msg.args[1],0);
	        else
	                msg.error(name(),"(wrong number of arguments)");
	        }
	else if (msg == "position")
		setPosition(msg);
	else if (msg == "orientation")
		setOrientation(msg);
	else if (msg == "size")
		setSize(msg);
	else
		ygNode::message(msg);
	}


void fltTransform::extractTransform(const ygString& source,const ygString& node,const int& mode)
	{
	ygNode *ygSource = ygNodeDB::find(source.c_str());
	if (ygSource) 
		{
		pfGroup* pointer;
		pfSCS *scsNode = NULL;
		pointer = (pfGroup*) ygSource->pfnode()->find(node.c_str(), pfGroup::getClassType());
		if (mode == 0)
			scsNode = (pfSCS*) pointer;
		else if (mode == 1)
			scsNode = (pfSCS*) pointer->getParent(0);
		else if (mode == 2)
			scsNode = (pfSCS*) pointer->getChild(0);
		if (scsNode) 
			{
			pfMatrix scs;
			scsNode->getMat(scs);
			setMatrix(scs);
			p_->sourcename = source;
			p_->sourcename += " ";
			p_->sourcename += node;
			p_->sourcename += " ";
			char modeChar[6];
			sprintf(modeChar,"%d",mode);
			p_->sourcename += modeChar;
			netKeyChanged("source");
			}
		else
	       printf("fltTransform: ERROR - node %s not found\n",node.c_str());
		}
	else
	    printf("fltTransform: ERROR - source node %s not found\n",source.c_str());
	}


void fltTransform::setPosition(const ygMessage& msg)
	{
	pfVec3 v;
	if (msg.getVec3Args(v))
		setPosition(v);
	else
		msg.error(name(),"(wrong number of arguments)");
	}


void fltTransform::setOrientation(const ygMessage& msg)
	{
	pfVec3 v;
	if (msg.getVec3Args(v))
		setOrientation(v);
	else
		msg.error(name(),"(wrong number of arguments)");
	}


void fltTransform::setSize(const ygMessage& msg)
	{
	pfVec3 v;
	if (msg.args.size() == 1)
		setSize(msg.floatArg(0));
	else if (msg.getVec3Args(v))
		setSize(v);
	else
		msg.error(name(),"(wrong number of arguments)");
	}


/***** Net functions *****/

void fltTransform::acceptNetKey(const ygString& key)
	{
	if (key == "matrix")
		setMatrix(p_->netMatrix);
	else if (key == "source")
		{
		int index = 0;
		ygString* source = p_->sourcename.nextToken(" ",&index);
		ygString* node = p_->sourcename.nextToken(" ",&index);
		ygString* mode = p_->sourcename.nextToken(" ",&index);
		extractTransform(*source,*node,atoi(mode->data()));
		delete source;
		delete node;
		delete mode;
		}
	else
		ygNode::acceptNetKey(key);
	}


void fltTransform::storeMatrixKey(void)
	{
	if (isNetMaster())
		{
		getMatrix(p_->netMatrix);
		netKeyChanged("matrix");
		}
	}
