//Description: loads geometry from a named node within an OpenFlight file
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> a source and node are specified
//<li> default source is called "fltSource"
//<li> default node is the name of the fltObject
//<li> desired node is found under source, removed, and given to the fltObject
//</ul>
//
// Category: Geometry
// Author: Alex Hill
// Revision: 11/01/01
//
#include <Performer/pfdu.h>
#include <Performer/pf/pfGroup.h>
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "ygNodeDB.h"
#include "ygPFObjectCache.h"
#include "fltObject.h"

extern "C" ygNode* construct_fltObject(const char* name,bool master) { return new fltObject(name,master); }

struct _fltObjectPrivateData
	{
	ygString filename;
	ygString sourcename;
	pfNode *obj;
	uint isectMask;
	bool drawn;
	};


fltObject::fltObject(const char* name,bool master) : ygTransform(name,master)
	{
	setClassName("fltObject",true);
	p_ = new struct _fltObjectPrivateData;
	p_->obj = NULL;
	p_->isectMask = YG_ISECT_ALLMASK;
	p_->drawn = true;
	addNetKey("file",&p_->filename,YG_NET_STRING);
	addNetKey("source",&p_->sourcename,YG_NET_STRING);
	addNetKey("isect",&p_->isectMask,YG_NET_INT);
	addNetKey("drawn",&p_->drawn,YG_NET_BOOL);
    pfdInitConverter("flt");
    pfdConverterMode("flt",1,FALSE);
	}


fltObject::~fltObject(void)
	{
	delete p_;
	}


void fltObject::reset(void)
	{
	p_->isectMask = YG_ISECT_ALLMASK;
	netKeyChanged("isect");
	if (p_->obj)
		{
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	ygTransform::reset();
	}


void fltObject::message(const ygMessage& msg)
	{
	//load the given case sensitive filename
	if (msg == "file")
		{
		if (msg.args.size() == 1)
			loadFile(msg.args[0],false);
		else if (msg.args.size() == 2)
			loadFile(msg.args[0],msg.boolArg(1));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//extract geometry with a given name from the default source node
	else if (msg == "node")
		{
		ygString fltSource("fltSource");
		if (msg.args.size() == 0)
			extractNode(fltSource,name(),0);
		else if (msg.args.size() == 1)
			extractNode(fltSource,msg.args[0],2);
		else if (msg.args.size() == 2)
			{
			if (msg.args[1] == "parent")
				extractNode(fltSource,msg.args[0],1);
			else
				extractNode(fltSource,msg.args[0],0);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//extract geometry with a given name from a specific source node
	else if (msg == "source")
		{
		if (msg.args.size() == 1)
			extractNode(msg.args[0],name(),0);
		else if (msg.args.size() == 2)
			extractNode(msg.args[0],msg.args[1],2);
		else if (msg.args.size() == 3)
			{
			if (msg.args[2] == "parent")
				extractNode(msg.args[0],msg.args[1],1);
			else
				extractNode(msg.args[0],msg.args[1],0);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set collision detection status
	else if (msg == "wall")
		{
		if (msg.args.size() > 0)
			setWall(msg.boolArg(0));
		else
			setWall();
		}
	//set walk navigation status
	else if (msg == "floor")
		{
		if (msg.args.size() > 0)
			setFloor(msg.boolArg(0));
		else
			setFloor();
		}
	//set if the object is drawn
	else if (msg == "draw")
		{
		if (msg.args.size() > 0)
			setDraw(msg.boolArg(0));
		else
			setDraw();
		}
	else
		ygTransform::message(msg);
	}


void fltObject::loadFile(const ygString& file,const bool& clean)
	{
	if (p_->obj)
		pfnode()->removeChild(p_->obj);
	if (!clean)
		pfdConverterMode("flt",2,FALSE);
	p_->obj = ygPFObjectCache::objectReference(file.c_str());
	if (p_->obj)
		{
		p_->filename = file;
		pfnode()->addChild(p_->obj);
		if (p_->drawn)
			p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			p_->obj->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask, PFTRAV_SELF|PFTRAV_DESCEND|PFTRAV_IS_CACHE, PF_SET);
		netKeyChanged("file");
		}
	}


void fltObject::extractNode(const ygString& source, const ygString& node, const int& mode)
        {
        if (p_->obj) 
                pfnode()->removeChild(p_->obj);
        ygNode *ygSource = ygNodeDB::find(source.c_str());
        if (ygSource)
			{
	        pfGroup* pointer;
                pointer = (pfGroup*) ygSource->pfnode()->find(node.c_str(), pfGroup::getClassType());
	        if (mode == 0)
	            p_->obj = pointer;
	        else if (mode == 1)
	            p_->obj = pointer->getParent(0);
	        else if (mode == 2)
	            p_->obj = pointer->getChild(0);
			if (p_->obj)
				{
				pfGroup *parent = (pfGroup*) p_->obj->getParent(0);
				parent->removeChild(p_->obj);
				}
			else
				printf("fltObject: ERROR - node %s not found\n",node.c_str());
			}
        else
			printf("fltObject: ERROR - source node %s not found\n",source.c_str());
        if (p_->obj)
			{
			if (p_->drawn)
				p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
			else
				p_->obj->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
			p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask, PFTRAV_SELF|PFTRAV_DESCEND|PFTRAV_IS_CACHE, PF_SET);
			pfnode()->addChild(p_->obj);
			p_->sourcename = source;
			p_->sourcename += " ";
			p_->sourcename += node;
			p_->sourcename += " ";
			char modeChar[6];
			sprintf(modeChar,"%d",mode);
			p_->sourcename += modeChar;
			netKeyChanged("source");
			}
        }


void fltObject::setWall(bool val)
	{
	if (val)
		p_->isectMask |= YG_ISECT_WALLMASK;
	else
		p_->isectMask &= ~YG_ISECT_WALLMASK;
	netKeyChanged("isect");
	if (p_->obj)
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}


void fltObject::setFloor(bool val)
	{
	if (val)
		p_->isectMask |= YG_ISECT_FLOORMASK;
	else
		p_->isectMask &= ~YG_ISECT_FLOORMASK;
	netKeyChanged("isect");
	if (p_->obj)
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}


void fltObject::setDraw(bool val)
	{
	p_->drawn = val;
	netKeyChanged("drawn");
	if (p_->obj)
		{
		if (p_->drawn)
			p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			p_->obj->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	}



void fltObject::acceptNetKey(const ygString& key)
	{
	if (key == "file")
		loadFile(p_->filename,false);
	else if (key == "source")
		{
		int index = 0;
		ygString* source = p_->sourcename.nextToken(" ",&index);
		ygString* node = p_->sourcename.nextToken(" ",&index);
		ygString* mode = p_->sourcename.nextToken(" ",&index);
		extractNode(*source,*node,atoi(mode->data()));
		delete source;
		delete node;
		delete mode;
		}
	else if (key == "drawn")
		setDraw(p_->drawn);
	else if (key == "isect")
		{
		if (p_->obj)
			p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	else
		ygTransform::acceptNetKey(key);
	}
