// Description: a clipping plane that can be positioned within the scene
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> the default plane will be the XY axis with the positive Z axis being clipped
//<li> the node message defines the node and its children that are affected
//<li> the affected node should neither be a transform nor a child of a transform
//<li> you must add the following to your RUN file in order to use this node:
//<ul>
//<li> setenv YG_PRELOAD_CLASSES clipPlane
//</ul>
//</ul>
//
// Category: Attributes
// Author: Alex Hill
// Revision: 03/25/02
//
#include <Performer/pf/pfGroup.h>
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include "clipPlane.h"

using namespace std;

extern "C" ygNode* construct_clipPlane(const char* name,bool master) { return new clipPlane(name,master); }

typedef struct _clipData
	{
	bool active;
	int plane;
	pfVec3 axis;
	float position;
	} clipData_t;

int clipPlane::planeNumber = -1;

clipPlane::clipPlane(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("clipPlane",true);
	//allocate shared memory clipData structure for pre and post callbacks
	data_ = (clipData_t*) pfMalloc(sizeof(clipData_t),pfGetSharedArena());
	//increment the plane number used by each clipPlane node (need test for max planes)
	planeNumber++;
	//set the plane number in the shared memory structure
	data_->plane = GL_CLIP_PLANE0+planeNumber;
	clipNode = NULL;
	data_->active = true;
	data_->axis.set(0,0,1);
	data_->position = 0.0;
	//distribute the name of the node affected
	addNetKey("clipName",&clipName,YG_NET_STRING);
	//distribute the on/off state of the clipping plane
	addNetKey("active",&data_->active,YG_NET_BOOL);
	}

clipPlane::~clipPlane(void)
	{
	}


void clipPlane::reset(void)
	{
	//clear the affected node name
	clipName.clear();
	clipNode = NULL;
	//set the plane to active
	data_->active = true;
	//set the axis to 0,0,1
	data_->axis.set(0,0,1);
	//set the distance to 0.0
	data_->position = 0.0;
	ygNode::reset();
	}


void clipPlane::message(const ygMessage& msg)
	{
	//set the name of the node to be affected
	if (msg == "node")
		{
		if (msg.args.size() > 0)
			{
			clipName = msg.args[0];
			setClipNode(clipName);
			netKeyChanged("clipName");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//turn the clipping plane on
	else if (msg == "on")
		{
		if (!data_->active)
			{
			data_->active = true;
			netKeyChanged("active");
			}
		}
	//turn the clipping plane off
	else if (msg == "off")
		{
		if (data_->active)
			{
			data_->active = false;
			netKeyChanged("active");
			}
		}
	else
		ygNode::message(msg);
	}

void clipPlane::setClipNode(const ygString& name)
	{
	//if an affected node is being clipped then
	if (clipNode)
		{
		//reset the traversal functions to NULL
		clipNode->pfnode()->setTravFuncs(PFTRAV_DRAW,NULL,NULL);
		clipNode->pfnode()->setTravData(PFTRAV_DRAW,NULL);
		}		
	ygNode* node = NULL;
	//find the desired affected node
	node = ygNodeDB::find(name);
	//if node is found then
	if (node)
		{
		clipNode = node;
		//set the pre and post draw traversal functions for affected pfNode
		clipNode->pfnode()->setTravFuncs(PFTRAV_DRAW,clipPreDraw,clipPostDraw);
		clipNode->pfnode()->setTravData(PFTRAV_DRAW,data_);
		}
	}
	
void clipPlane::app(void)
	{
	if (data_->active)
		{
		//get current node transformation
		pfMatrix matrix;
		getTransform(matrix);
		//get the position from the transformation
		pfVec3 position;
		position.set(0,0,0);
		position.xformPt(position,matrix);
		//transform a unit vector by the transformation
		pfVec3 normal;
		normal.set(0,0,1);
		normal.xformPt(normal,matrix);
		//get the normal by subtracting the position from unit vector
		normal -= position;
		//normalize the normal
		normal.normalize();
		//set the plane equation to the three values of the normal vector
		data_->axis[0] = normal[0];
		data_->axis[1] = normal[1];
		data_->axis[2] = normal[2];
		//calculate the dot product of the normal and the position vector
		float distance;
		distance = -position.dot(normal);
		//set the distance to the plane equal to the dot product
		data_->position = distance;
		}
	ygNode::app();
	}

int clipPlane::clipPreDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the clipData structure
	clipData_t* clipData = (clipData_t*)data;
	//if the plane is active then
	if (clipData->active)
		{
		//construct an array of size 4
		GLdouble plane[4];
		//set array to hold plane equation
		plane[0] = clipData->axis[0];
		plane[1] = clipData->axis[1];
		plane[2] = clipData->axis[2];
		plane[3] = clipData->position;
		//enable clipping plane indicated by plane number
		glEnable(clipData->plane);
		//call glClipPlane to set the plane equation
		glClipPlane(clipData->plane, plane);
		}
	return PFTRAV_CONT;
	}

int clipPlane::clipPostDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the clipData structure
	clipData_t* clipData = (clipData_t*)data;
	//disable the clipping plane indicated by plane number
	glDisable(clipData->plane);
	return PFTRAV_CONT;
	}

void clipPlane::acceptNetKey(const ygString& key)
	{
	if (key == "clipName")
		setClipNode(clipName);
	else
		ygNode::acceptNetKey(key);
	}

