// Description: a transform that orients to face the viewer about a point or axis
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the model should be oriented with the negative Y axis
//<li> the point mode rotates the model about a point
//<li> the optional axis is the model up vector in point mode
//<li> the axis mode rotates the model about an axis
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/01
//
#include <Performer/pf/pfBillboard.h>
#include <ygUser.h>
#include <ygWorld.h>
#include "billboard.h"

extern "C" ygNode* construct_billboard(const char* name,bool master) { return new billboard(name,master); }

billboard::billboard(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("billboard",true);
	axis.set(0,0,1);
	mode = PFBB_AXIAL_ROT;
	dcs = new pfDCS;
	setPfNode(dcs);
	}

void billboard::reset(void)
	{
	//set the rotation mode to axis
	mode = PFBB_AXIAL_ROT;
	//set the rotation axis to the Z-axis (0,0,1)
	axis.set(0,0,1);
	ygNode::reset();
	}


void billboard::message(const ygMessage& msg)
	{
	//set the mode and optional axis
	if (msg == "mode")
		{
		if (msg.args.size() > 0)
			{
			mode = PFBB_AXIAL_ROT;
			if (msg.args[0] == "point")
				mode = PFBB_POINT_ROT_EYE;
			else if (msg.args[0] == "axis")
				mode = PFBB_AXIAL_ROT;
			if (msg.args.size() > 1)
				{
				msg.getVec3Args(axis,1);
				axis.normalize();
				}
			else
				axis.set(0,0,1);
			}
		else
			{
			mode = PFBB_AXIAL_ROT;
			axis.set(0,0,1);
			}
		}
	else
		ygNode::message(msg);
	}


void billboard::app(void)
	{
	//find viewer in the world
	ygNode* viewer = ygWorld::World->viewer();
	//if a viewer is found
	if (viewer) 
		{
		//get viewer position vector relative to this node
		pfVec3 viewerVector = viewer->origin(parent()); 
		viewerVector.normalize();

		//get the cross product of the viewer vector and the rotation axis
		pfVec3 viewCross;
		viewCross.cross(axis,viewerVector);
		viewCross.normalize();

		//establish the object front as the negative Y-axis
		pfVec3 frontAxis(0,-1,0);
		pfVec3 frontCross;
		//get the cross product of the front axis and the rotation axis
		frontCross.cross(axis,frontAxis);
		frontCross.normalize();
		float adjacent = frontCross.dot(viewCross);
		pfVec3 thirdNormal;
		//get a normal towards the viewer by crossing this cross with the rotaion axis
		thirdNormal.cross(axis,frontCross);
		float opposite = thirdNormal.dot(viewCross);
		//get the rotation to the viewer vector
		float axisRot = pfArcTan2(opposite,adjacent);
		//create a matrix with this rotation
		pfMatrix matrix;
		matrix.makeRot(axisRot,axis[0],axis[1],axis[2]);

		//if the mode if point then
		if (mode == PFBB_POINT_ROT_EYE)
			{
			float crossRot;
			//get the rotation about the viewer cross product
			crossRot = pfArcSin(axis.dot(viewerVector));
			//set the final matrix to this rotation
			pfMatrix pitchMat;
			pitchMat.makeRot(-crossRot,viewCross[0],viewCross[1],viewCross[2]);
			//add this rotation matrix to the existing rotation matrix
			matrix.postMult(pitchMat);
			}

		//set the dcs matrix
		dcs->setMat(matrix);
		}
	ygNode::app();
	}
