// Description: applies a texture to the existing texture on objects below it
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> a texture of any size can be applied to the manipulated texture
//<li> a texture position of -0.5 places the center of the applied texture at the lower extreme boundary
//<li> a texture position of 0.5 places the center of the applied texture at the upper extreme boundary
//<li> both the texture being manipulated and the applied texture must have the same number of components
//<li> the blend mode replaces applied texture pixels with pixels from the manipulated texture for alpha values above the threshold
//<li> the clean mode automatically restores the last application before doing the next apply
//<li> you must add the following to your RUN file in order to use this node:
//<ul>
//<li> setenv YG_PRELOAD_CLASSES applyTexture
//</ul>
//</ul>
//
// Category: Attributes
// Author: Alex Hill
// Revision: 07/10/02
//
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pf/pfGroup.h>
#include <ygNetKeys.h>
#include "applyTexture.h"

extern "C" ygNode* construct_applyTexture(const char* name,bool master) { return new applyTexture(name,master); }

typedef struct _applyData
	{
	unsigned char* srceImage;
	int xSrcePos;
	int ySrcePos;
	int srceWidth;
	int xDestPos;
	int yDestPos;
	int xSrceSize;
	int ySrceSize;
	pfTexture* texture;
	bool reset;
	bool restore;
	bool clean;
	bool apply;
	} applyData_t;

applyTexture::applyTexture(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("applyTexture",true);
	destImage = NULL;
	data_ = (applyData_t*) pfMalloc(sizeof(applyData_t),pfGetSharedArena());
	pfnode()->setTravFuncs(PFTRAV_DRAW,applyPreDraw,NULL);
	pfnode()->setTravData(PFTRAV_DRAW,data_);
	srceImage = NULL;
	gstate = NULL;
	texture = NULL;
	blend = false;
	data_->reset = false;
	data_->clean = false;
	threshold = 255;
	position[0] = 0.0;
	position[1] = 0.0;
	data_->texture = NULL;
	data_->restore = false;
	data_->apply = false;
	srceXSize = 0;
	srceYSize = 0;
	srceNumComp = 0;
	destXSize = 0;
	destYSize = 0;
	destNumComp = 0;
	subtexture = new pfTexture;
	//distribute the position to apply the texture
	addNetKey("apply",&position,YG_NET_VEC2);
	//distribute the name of the texture to apply
	addNetKey("file",&filename,YG_NET_STRING);
	//distribute the mode of texture restore
	addNetKey("restore",&data_->restore,YG_NET_BOOL);
	//distribute the reset mode
	addNetKey("reset",&data_->reset,YG_NET_BOOL);
	//distribute the blend mode
	addNetKey("blend",&blend,YG_NET_BOOL);
	//distribute the blend threshold
	addNetKey("thresh",&threshold,YG_NET_INT);
	//distribute the clean mode
	addNetKey("clean",&data_->clean,YG_NET_BOOL);
	}

applyTexture::~applyTexture(void)
	{
	}

void applyTexture::reset(void)
	{
	//set blend mode to false
	blend = false;
	netKeyChanged("blend");
	//set threshold to 255
	threshold = 255;
	netKeyChanged("thresh");
	//set X and Y position to 0.0
	position[0] = 0.0;
	position[1] = 0.0;
	netKeyChanged("apply");
	//clear the filename strings
	filename.clear();
	netKeyChanged("file");
	//initialize source size and components
	srceNumComp = 0;
	//initialize destination size and components
	destNumComp = 0;
	//set reset mode to true
	data_->reset = true;
	netKeyChanged("reset");
	ygNode::reset();
	}


void applyTexture::message(const ygMessage& msg)
	{
	//reset texture to state before last apply or clean
	if (msg == "restore")
		{
		data_->restore = true;
		netKeyChanged("restore");
		}
	//load the given texture file to apply
	else if (msg == "file")
		{
		if (msg.args.size() == 1)
			loadFile(msg.stringArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//apply texture file at position
	else if (msg == "apply")
		{
		if (msg.args.size() == 2)
			{
			pfVec2 pos;
			msg.getVec2Args(pos);
			apply(pos);
			}
		else
			apply(position);
		}
	//set position to apply the texture
    else if (msg == "position")
		{
		if (msg.args.size() == 2)
			msg.getVec2Args(position);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the blend mode
	else if (msg == "blend")
		{
		if (msg.args.size() > 0)
			blend = msg.boolArg(0);
		else
			blend = false;
		netKeyChanged("blend");
		}
	//set the blend threshold
	else if (msg == "threshold")
		{
		if (msg.args.size() > 0)
			threshold = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		netKeyChanged("thresh");
		}
	//set the clean mode
	else if (msg == "clean")
		{
		if (msg.args.size() > 0)
			data_->clean = msg.boolArg(0);
		else
			data_->clean = false;
		netKeyChanged("clean");
		}
	else
		ygNode::message(msg);
	}

void applyTexture::loadFile(const ygString& file)
	{
	//if filename loaded correctly then
	if (subtexture->loadFile(file.c_str()))
		{
		//get image size
		subtexture->getImage((uint**)&srceImage,&srceNumComp,&srceXSize,&srceYSize,NULL);
		//set the filename net key
		filename = file;
		//indicate a change in the filename net key
		netKeyChanged("file");
		}
	}

void applyTexture::apply(const pfVec2& pos)
	{
	//if texture has been found and number of components are the same then
	if (texture && srceNumComp > 0 && srceNumComp == destNumComp)
		{
		position = pos;
		//initialize source image data and texture object
		data_->srceImage = srceImage;
		data_->texture = texture;
		//set apply flag to true
		data_->apply = true;
		//set restore flag to true if clean is true
		if (data_->clean)
			data_->restore = true;
		data_->srceWidth = srceXSize;
		//set the position and size of the texture to be loaded
		data_->xDestPos = (0.5 + position[0])*destXSize - srceXSize/2;
		data_->xSrcePos = 0;
		data_->xSrceSize = srceXSize;
		if (data_->xDestPos > destXSize || data_->xDestPos < -srceXSize)
			{
			data_->xDestPos = 0;
			data_->xSrceSize = 0;
			}
		else if (data_->xDestPos < 0)
			{
			data_->xSrcePos = -data_->xDestPos;
			data_->xDestPos = 0;
			data_->xSrceSize = srceXSize - data_->xSrcePos;
			}
		else if (data_->xDestPos > destXSize - srceXSize)
			{
			data_->xSrceSize = destXSize - data_->xDestPos;
			}
		data_->yDestPos = (0.5 + position[1])*destYSize - srceYSize/2;
		data_->ySrcePos = 0;
		data_->ySrceSize = srceYSize;
		if (data_->yDestPos > destYSize || data_->yDestPos < -srceYSize)
			{
			data_->yDestPos = 0;
			data_->ySrceSize = 0;
			}
		else if (data_->yDestPos < 0)
			{
			data_->ySrcePos = -data_->yDestPos;
			data_->yDestPos = 0;
			data_->ySrceSize = srceYSize - data_->ySrcePos;
			}
		else if (data_->yDestPos > destYSize - srceYSize)
			{
			data_->ySrceSize = destYSize - data_->yDestPos;
			}
		//if number of components if 4 and blend flag is true
		if (srceNumComp == 4 && blend)
			{
			int xDestPos,yDestPos;
			int xSrcePos,ySrcePos;
			int xSrceEnd,ySrceEnd;
			xSrceEnd = data_->xSrcePos + data_->xSrceSize;
			ySrceEnd = data_->ySrcePos + data_->ySrceSize;
			yDestPos = data_->yDestPos;
			//for each Y element in source image do
			for (ySrcePos=data_->ySrcePos;ySrcePos<ySrceEnd;ySrcePos++)
				{
				xDestPos = data_->xDestPos;
				//for each X element in source image do
				for (xSrcePos=data_->xSrcePos;xSrcePos<xSrceEnd;xSrcePos++)
					{
					//if alpha value is at or above threshold then
					if (srceImage[(ySrcePos*srceXSize + xSrcePos)*srceNumComp+3] >= threshold)
						{
						//update source image with corresponding destination image RGB values
						srceImage[(ySrcePos*srceXSize + xSrcePos)*srceNumComp] = destImage[(yDestPos*destXSize + xDestPos)*destNumComp];
						srceImage[(ySrcePos*srceXSize + xSrcePos)*srceNumComp+1] = destImage[(yDestPos*destXSize + xDestPos)*destNumComp+1];
						srceImage[(ySrcePos*srceXSize + xSrcePos)*srceNumComp+2] = destImage[(yDestPos*destXSize + xDestPos)*destNumComp+2];
						}
					xDestPos++;
					}
				yDestPos++;
				}
			}
		//indicate a change in the apply net key
		netKeyChanged("apply");
		}
	}
	
int applyTexture::applyPreDraw(pfTraverser*,void *data)
	{
	static int lastXDestPos;
	static int lastYDestPos;
	static int lastXSrceSize;
	static int lastYSrceSize;
	//retrieve callback data
	applyData_t* applyData = (applyData_t*)data;
	//if reset flag is true then
	if (applyData->reset)
		{
		//restore whole destination image
		if (applyData->texture)
			applyData->texture->load();
		applyData->reset = false;
		applyData->restore = false;
		applyData->texture = NULL;
		applyData->clean = false;
		applyData->restore = false;
		applyData->apply = false;
		}
	//if restore flag is true then
	if (applyData->restore)
		{
		uint* destImage;
		int srceWidth;
		//get destination texture image data and width
		applyData->texture->getImage(&destImage,NULL,&srceWidth,NULL,NULL);
		//if clean is true then restore whole destination image
		if (applyData->clean)
			applyData->texture->load();
		//esle, restore last position and size
		else
			applyData->texture->subload(PFTEX_SOURCE_IMAGE,destImage,lastXDestPos,lastYDestPos,srceWidth,lastXDestPos,lastYDestPos,lastXSrceSize,lastYSrceSize);
		//reset restore flag to false
		applyData->restore = false;
		}
	//if apply flag is true then
	if (applyData->apply)
		{
		//store destination position and size for later restoration
		lastXDestPos = applyData->xDestPos;
		lastYDestPos = applyData->yDestPos;
		lastXSrceSize = applyData->xSrceSize;
		lastYSrceSize = applyData->ySrceSize;
		//apply source texture at given position and size
		//applyData->texture->subload(PFTEX_SOURCE_IMAGE,(uint*)applyData->srceImage,applyData->xSrcePos,applyData->ySrcePos,applyData->srceWidth,applyData->xDestPos,applyData->yDestPos,applyData->xSrceSize,applyData->ySrceSize);
		applyData->texture->subloadLevel(PFTEX_SOURCE_IMAGE,(uint*)applyData->srceImage,applyData->xSrcePos,applyData->ySrcePos,applyData->srceWidth,applyData->xDestPos,applyData->yDestPos,applyData->xSrceSize,applyData->ySrceSize,PFTEX_LEVEL_ALL);
		//reset apply flag to false
		applyData->apply = false;
		}
	return PFTRAV_CONT;
	}
	
void applyTexture::setChildrenTextures(pfNode *node)
	{
	int i;
	//if node type is pfGeode then set geoset textures
	if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode *)node;
		//for each geode, call setGeosetTexture
		for (i=0; i < geode->getNumGSets(); i++)
			setGeosetTexture(geode->getGSet(i));
		}
	//else, if node type is pfGroup
	else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		//call setChildrenTextures recursively on all children
		for (i=0; i < group->getNumChildren(); i++)
			setChildrenTextures(group->getChild(i));
		}
	}

void applyTexture::setGeosetTexture(pfGeoSet *gset)
	{
	//get the geoset geostate
	gstate = gset->getGState();
	//if the geostate has texturing enabled
	if (gstate->getMode(PFSTATE_ENTEXTURE) == PF_ON)
		{
		//set texture node to the geostate texture 
		texture = (pfTexture*)gstate->getAttr(PFSTATE_TEXTURE);
		//configure the texture to allow subloading
		texture->setFormat(PFTEX_SUBLOAD_FORMAT,PF_ON);
		//retrieve the texture image data, size, and number of components
		texture->getImage((uint**)&destImage,&destNumComp,&destXSize,&destYSize,NULL);
		}
	}
	
void applyTexture::app(void)
	{
	//if first frame then find a texture node from a child node
	if (pfChanged())
		setChildrenTextures(pfnode());
	ygNode::app();
	}
	
void applyTexture::acceptNetKey(const ygString& key)
	{
	if (key == "file")
		{
		if (filename.length() > 0)
			loadFile(filename);
		else
			{		
			srceNumComp = 0;
			destNumComp = 0;
			}
		}
	else if (key == "apply")
		apply(position);
	else
		ygNode::acceptNetKey(key);
	}
