// Description: a node that generates events with the value of the microphone amplitude
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Sound
// Author: Alex Hill
// Revision: 08/01/03
//
#include <pfcave.h>
#include <bergenAmplitude.h>
#include <ygWorld.h>
#include <ygNetKeys.h>
#include "amplitude.h"

using namespace std;

extern "C" ygNode* construct_amplitude(const char* name,bool master) { return new amplitude(name,master); }

struct _amplitudePrivateData
	{
	bergenAmplitude * sample;
	bool isPlaying;
	float inputAmplitude;
	float threshold;
	float filter;
	float curAmplitude;
	pfVec3 currPosition;
	float updateInterval, lastUpdate;
	};


amplitude::amplitude(const char* name,bool master) : ygSoundServer(name,master)
	{
	setClassName("amplitude");
	p_ = new struct _amplitudePrivateData;
	p_->sample = NULL;
	p_->isPlaying = false;
	p_->threshold = 0.2;
	p_->inputAmplitude = 0.0;
	p_->filter = 0.9;
	p_->updateInterval = 0.05;
	p_->lastUpdate = 0;
	}

amplitude::~amplitude(void)
	{
	if (p_->sample)
		delete p_->sample;
	delete p_;
	}


void amplitude::reset(void)
	{
	//reset playing flag to false
	p_->isPlaying = false;
	//set current amplitude to 0.0
	p_->inputAmplitude = 0.0;
	//set threshold to 0.2
	p_->threshold = 0.2;
	//set filter to 0.9
	p_->filter = 0.9;
	//set update interval to 0.05
	p_->updateInterval = 0.05;
	//if a sample exists then delete it
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	ygSoundServer::reset();
	}


void amplitude::message(const ygMessage& msg)
	{
	//set the amplitude within the space
	if (msg == "amplitude")
		{
		if (msg.args.size() > 0)
			setAmplitude(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the zero amplitude distance from the space perimeter
	else if (msg == "falloffDistance")
		{
		if (msg.args.size() > 0)
			setFalloffDistance(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the minimum amplitude threshold
	else if (msg == "threshold")
		{
		if (msg.args.size() > 0)
			p_->threshold = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the first order filter constant (percent contribution)
	else if (msg == "filter")
		{
		if (msg.args.size() > 0)
			p_->filter = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//start monitoring the input signal with optional amplitude
	else if (msg == "play")
		{
		if (msg.args.size() > 0)
			{
			setAmplitude(msg.floatArg(0));
			play();
			}
		else
			play();
		}
	//stop the input monitoring
	else if (msg == "stop")
		stop();
	//set the update interval for sound server
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSoundServer::message(msg);
	}


void amplitude::setCurAmplitude(float val)
	{
	p_->curAmplitude = val;
	if (p_->sample)
		p_->sample->setAmplitude(val);
	}


bool amplitude::isPlaying(void)
	{
	return p_->isPlaying;
	}


void amplitude::play(void)
	{
	if (p_->isPlaying)
		return;
	//if no sample then create a new bergenAmplitude
	if (!p_->sample)
		p_->sample = new bergenAmplitude(server());
	//play the sample
	p_->sample->play();
	//generate a heartbeat for the server
	heartbeat();
	p_->isPlaying = true;
	eventOccurred("start");
	}


void amplitude::stop(bool sendStopCommand)
	{
	//if a sample is playing
	if (p_->isPlaying)
		{
		//send a stop command to the server
		if ((sendStopCommand) && (p_->sample))
			p_->sample->stop();
		p_->isPlaying = false;
		eventOccurred("stop");
		}
	}


void amplitude::app(void)
	{
	//if a sample exists generate a heartbeat for the server
	if (p_->sample)
		{
		//generate a heartbeat for the server
		heartbeat();
		//if the sample is playing
		if (p_->isPlaying)
			{
			//if update interval has passed then update sound server
			if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
				{
				float currAmplitude;
				currAmplitude = p_->sample->amplitude();
				if (currAmplitude < p_->threshold)
					currAmplitude = 0.0;
				currAmplitude = currAmplitude*p_->filter + p_->inputAmplitude*(1-p_->filter);
				if (currAmplitude != p_->inputAmplitude)	
					{
					p_->inputAmplitude = currAmplitude;
					ygString args("value=");
					char val[32];
					sprintf(val,"%f",p_->inputAmplitude);
					args += val;
					//generate changed event with $value
					eventOccurred("changed",args);
					}
				float newAmpl = computeAmplitude();
				if (fabs(newAmpl - p_->curAmplitude) > 0.005)
					setCurAmplitude(newAmpl);
				//determine position of this sound
				pfVec3 newPos;
				newPos = origin();
				pfVec3 cavePos;
				//get position relative to CAVE
				CAVENavConvertWorldToCAVE(newPos.vec,cavePos.vec);
				if (cavePos != p_->currPosition)
					{
					if (p_->sample)
						p_->sample->setPosition(cavePos.vec[0],cavePos.vec[2],-cavePos.vec[1]);
					p_->currPosition = cavePos;
					}
				p_->lastUpdate = ygWorld::FrameTime;
				}
			}
		}
	ygSoundServer::app();
	}
