//Description: class used by the MovieSurface node
//
//<b>notes:</b>
//<ul>
//<li> this node currently is supported only on LINUX machines
//<li> for information about xine, click <a href="http://www.evl.uic.edu/eleni/yg/xine.html"> here.</a>
//<li> enables the xine multimedia engine
//</ul>
//
//
// Category: None
// Author: Kalle Jalkanen and Jonas Westling
//         Interactive Institute Ume, Sweden
// Revision: Helen-Nicole Kostis - documentation
//


#include "MoviePlayer.h"


MoviePlayer::MoviePlayer()
{
	xine = NULL;
	stream = NULL;
	vo_port = NULL;
	ao_port = NULL;
	event_queue = NULL;
	vo_frame_ptr = NULL;
	ao_frame_ptr = NULL;	
	running = 0;
	rgb = NULL;
	width = 0;
	height = 0;
	start_vpts = 0;
	xine = xine_new();
	xine_init(xine);
	vo_port = xine_new_framegrab_video_port( xine );
 	ao_port = xine_new_framegrab_audio_port( xine );   
}



MoviePlayer::~MoviePlayer()
{

  	xine_close_video_driver(xine, vo_port); 
	xine_close_audio_driver(xine, ao_port); 
   	xine_exit(xine);

	delete [] rgb;
}


int MoviePlayer::open(char *moviefile)
{

	stream = xine_stream_new(xine, ao_port, vo_port);
 	event_queue = xine_event_new_queue(stream);
	
	if (rgb) delete [] rgb;
	rgb = NULL;
	
	if (!xine_open(stream, moviefile))
	{
		printf("Unable to open mrl '%s'\n", moviefile);
		return 0;
   	} 

	width = xine_get_stream_info(stream, XINE_STREAM_INFO_VIDEO_WIDTH);
   	height = xine_get_stream_info(stream, XINE_STREAM_INFO_VIDEO_HEIGHT);
	
	rgb = new unsigned char[width*height*3];
	memset(rgb, 0, width*height*3);  

	start_vpts = 0;
	running = 0;	
	return 1;
}



int MoviePlayer::play(int start_pos, int start_time)
{
  	
	if (!xine_play(stream, start_pos, start_time)) 
	{
     		 printf("Unable to play mrl stream\n");
     		 return 0;
   	}

   	gettimeofday(&starttime, NULL);
  	 running = 1;


	int val = xine_get_next_video_frame(vo_port, &vo_frame);
	if (val) {
		xine_free_video_frame(vo_port, &vo_frame);
		start_vpts = vo_frame.vpts;
	}
	
	return 1;
}


int MoviePlayer::stop()
{
	if (vo_frame_ptr) 
	{
		xine_free_video_frame(vo_port, vo_frame_ptr);
		vo_frame_ptr = NULL;
	}

	xine_stop(stream);
	start_vpts = 0;
	running = 0;
	return 1;
}


int MoviePlayer::close()
{
	xine_close(stream);

	xine_event_dispose_queue(event_queue);
  	xine_dispose(stream);
	
	return 1;
}



void yuv420p2rgb(unsigned char *buf, unsigned char *rgb, int width, int height)
{
	int C, Y, U, V;
	int i, x, y;
	int w, h;
	
	unsigned char *pY, *pU, *pV;
	
	w = width;
	h = height;

	for (y = 0; y < h; ++y) {
		pY = buf + w*y;
		pU = buf + w*h + ((y>>1) * (w>>1));
		pV = buf + ((5*w*h)>>2) + ((y>>1) * (w>>1));
	
		for (x = 0; x < w; ++x) {
			Y = (pY[x] - 16) * 1192;
         U = pU[x>>1] - 128;            
         V = pV[x>>1] - 128;
			
			i = (x + w*y) * 3;
		
			// Red	
         C = Y + 1634 * V;
         C = C < 0 ? 0 : C;
         rgb[i+2] = C > 261120 ? 255 : (C>>10);
			
			// Green
			C = Y - 833 * V - 400 * U;
         C = C < 0 ? 0 : C;
         rgb[i+1] = C > 261120 ? 255 : (C>>10);
			
			// Blue
			C = Y + 2066 * U;
			C = C < 0 ? 0 : C;
         rgb[i+0] = C > 261120 ? 255 : (C>>10);
		}
	
	}
}




void yuv422i2rgb(unsigned char *buf, unsigned char *rgb, int width, int height)
{
	register int C, Y, U, V;
	int i, x, y;
	int w, h;
	
	unsigned char *p;
	
	w = width;
	h = height;

	for (y = 0; y < h; ++y) {
		p = buf + 2*w*y;
      
		for (x = 0; x < w; ++x) {
			Y = (p[x<<1] - 16) * 1192;
         
         U = p[4*(x>>1)+1] - 128;            
         V = p[4*(x>>1)+3] - 128;
			
			i = (x + w*y) * 3;

			// Red	
         C = Y + 1634 * V;
         C = C < 0 ? 0 : C;
         rgb[i] = C > 261120 ? 255 : (C>>10);
				
			// Green
			C = Y - 833 * V - 400 * U;
         C = C < 0 ? 0 : C;
         rgb[i+1] = C > 261120 ? 255 : (C>>10);
			
			// Blue
			C = Y + 2066 * U;
			C = C < 0 ? 0 : C;         
         rgb[i+2] = C > 261120 ? 255 : (C>>10);

		}
	
	}
}


double timediff(struct timeval *st, struct timeval *et)
{
  	struct timeval t;

   	t.tv_sec = et->tv_sec - st->tv_sec;
   
   	if (et->tv_usec < st->tv_usec) 
	{
 	    	t.tv_sec--;
  	    	t.tv_usec = 1000000L - st->tv_usec + et->tv_usec;
   	} 
	else 
	{
 	     	t.tv_usec = et->tv_usec - st->tv_usec;
   	}
   	return ((double)t.tv_sec + (double)t.tv_usec*1e-6);
}




void MoviePlayer::event_listener(void *user_data, const xine_event_t *event) 
{
	switch(event->type) 
	{
		case XINE_EVENT_UI_PLAYBACK_FINISHED:
         	running = 0;
		stop();
         	break;

      		case XINE_EVENT_PROGRESS:
         	{
            		xine_progress_data_t *pevent = (xine_progress_data_t *) event->data;
			printf("%s [%d%%]\n", pevent->description, pevent->percent);
         	}
         	break;
      		default:
         	printf("Event received: %d\n", event->type);

   	}
}



int MoviePlayer::app()
{
	if (running) 
	{
		double diff;
		int64_t loop_vpts, frame_vpts;
		int got_vo_frame=0;
		xine_event_t *event;
		event = xine_event_get( event_queue );
		if (event) 
		{
			event_listener(NULL, event);   
			xine_event_free(event);
		}

		if (!running) return 0;

		gettimeofday(&currtime, NULL);
		diff = timediff(&starttime, &currtime);
		loop_vpts = (int64_t)(90000.0*diff);
 
		while (1) 
		{
			if (vo_frame_ptr)
				xine_free_video_frame(vo_port, vo_frame_ptr);
			
			got_vo_frame = xine_get_next_video_frame(vo_port, &vo_frame);
			frame_vpts = vo_frame.vpts - start_vpts;
			if (got_vo_frame) vo_frame_ptr = &vo_frame;
			else vo_frame_ptr = NULL;

			if (got_vo_frame && (frame_vpts < loop_vpts)) 
			{

			} 
			else 
			{
				break;  
			}
		} 



		if (got_vo_frame && rgb) {
			
			if (vo_frame.colorspace == XINE_IMGFMT_YV12) {
				yuv420p2rgb(vo_frame.data, rgb, vo_frame.width, vo_frame.height);
			} else if (vo_frame.colorspace == XINE_IMGFMT_YUY2) {
				yuv422i2rgb(vo_frame.data, rgb, vo_frame.width, vo_frame.height);
			}

			if ( (frame_vpts - loop_vpts) > 0) {
				struct timespec sleepy;
				float nsecs = ((float)(frame_vpts-loop_vpts)/90000.0 *1.0E9);
				sleepy.tv_sec = (long)(nsecs/1.0E9);
				sleepy.tv_nsec = (long)(nsecs - sleepy.tv_sec*1.0E9);
				nanosleep( &sleepy, NULL );
			}
		}



		return 1;	
	}
	return 0;
}




