// Description: replaces the texture on objects below it with that of a movie file
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> the Quicktime (*.mov) photoJPEG format is only supported on Linux and requires the openquicktime library
//<li> you must add the following to your RUN file in order to use this node on Linux:
//<ul>
//<li> setenv LD_LIBRARY_PATH /usr/local/lib
//</ul>
//</ul>
//
// Category: Attributes
// Author: Dave Pape
//           11/10/01
// Revision: 01/10/03 Alex Hill - added Linux support thanks to Ben Chang
//
#include <malloc.h>
#include <Performer/pf/pfSwitch.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfTexture.h>
#include "ygUtil.h"
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "movieTexture.h"
#include "movieObject.h"


using namespace std;

extern "C" ygNode* construct_movieTexture(const char* name,bool master) { return new movieTexture(name,master); }


struct _movieTexturePrivateData
	{
	pfTexture* texture;
	bool active;
	float startTime;
	int width;
	int height;
	float rate;
	unsigned char* buffer; 
	movieObject movie;
	ygString netFile;
	bool debugFrame;
	bool loop;
	};

movieTexture::movieTexture(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("movieTexture",true);
	p_ = new struct _movieTexturePrivateData;
	p_->active = false;
	p_->texture = new pfTexture;
	addNetKey("file",&p_->netFile,YG_NET_STRING);
	addNetKey("active",&p_->active,YG_NET_BOOL);
	debugFlag("frame",&p_->debugFrame);
	p_->loop = true;
	p_->movie.setLoop(p_->loop);
	}

movieTexture::~movieTexture(void)
	{
	delete p_;
	}


void movieTexture::reset(void)
	{
	ygNode::reset();
	//set active to false
	p_->active = false;
	//turn off frame debug
	p_->debugFrame = false;
	netKeyChanged("active");
	}


void movieTexture::message(const ygMessage& msg)
	{
	//set the name of the movie file
	if (msg == "movie")
		{
		if (msg.args.size() == 1)
			setMovie(msg.args[0]);
		else
			msg.error(name(), "(wrong number of arguments)");
		}
	//start the movie from the beginning
	else if (msg == "play")
		{
		p_->active = true;
		p_->startTime = ygWorld::FrameTime;
		netKeyChanged("active");
		}
	else if (msg=="loop")
		{
		p_->loop = msg.intArg(0);
		p_->movie.setLoop(p_->loop);
		}
	else if (msg=="key")
		{
		p_->movie.setChromaKey(msg.intArg(0));
		}
	else if (msg=="keycolor")
		{
		p_->movie.setChromaKey(true);
		p_->movie.setChromaKeyColor(msg.floatArg(0),msg.floatArg(1),msg.floatArg(2));
		}
	else if (msg=="keytolerance")
		{
		p_->movie.setChromaKeyTolerance(msg.floatArg(0));
		}
	else if (msg=="keyfuzziness")
		{
		p_->movie.setChromaKeyFuzziness(msg.floatArg(0));
		}
	else
		ygNode::message(msg);
	}


void movieTexture::setMovie(const ygString& file)
	{
	char *searchPath = ".";
	ygString fullPath;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: movieTexture::show: could not find file '%s'\n",file.c_str());
		return;
		}
	if (p_->movie.isOpen())
		p_->movie.close();
	printf ("opening movie: %s\n",fullPath.c_str());
	p_->movie.open(fullPath.c_str());
	if (!p_->movie.isOpen())
		{
		fprintf(stderr,"ERROR (movieTexture::show): failed to open '%s'\n",fullPath.c_str());
		return;
		}
	p_->width = p_->movie.getWidth();
	p_->height = p_->movie.getHeight();
	p_->rate = p_->movie.getRate();
	p_->buffer = (unsigned char*)pfMalloc(p_->width*p_->height*4*sizeof(unsigned char),pfGetSharedArena());
	p_->netFile = file;
	netKeyChanged("file");
	}

void movieTexture::app(void)
	{
	if (p_->active)
		{
		if (pfChanged())
			replaceTextures(pfnode());
		int frame = (ygWorld::FrameTime - p_->startTime)*p_->rate;		
		if (p_->debugFrame)
			cout << "Reading frame " << frame << " of " << p_->netFile << endl;
		if (p_->movie.getFrame(frame,p_->buffer))
			{
			p_->texture->setFormat(PFTEX_IMAGE_FORMAT, GL_ABGR_EXT);
			p_->texture->setImage((uint*)p_->buffer,4,p_->width,p_->height,1);
			}
		else
			{
			p_->active = false;
			netKeyChanged("active");
			eventOccurred("end");
			}
		}
	ygNode::app();
	}

void movieTexture::replaceTextures(pfNode * node)
	{
	int i;
	if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode *)node;
		for (i=0; i < geode->getNumGSets(); i++)
			{
			pfGeoSet *gset = geode->getGSet(i);
			gset->getGState()->setAttr(PFSTATE_TEXTURE, p_->texture);
			}
		}
	else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		for (i=0; i < group->getNumChildren(); i++)
			replaceTextures(group->getChild(i));
		}
	}

void movieTexture::acceptNetKey(const ygString& key)
	{
	if (key == "file")
		{
		setMovie(p_->netFile);
		}
	else if (key == "active")
		{
		if (p_->active == true)
			p_->startTime = ygWorld::FrameTime;
		}
	else
		ygNode::acceptNetKey(key);
	}
	
	
