// Description: a node that reads data from a Maya Animation Curve Export File
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//</ul>
//
// Category: Transformation
// Author: Ben Chang
// Revision: 09/15/03
//

#include <ygUtil.h>
#include <ygWorld.h>
#include "mayaAnimation.h"
#include "ygNetKeys.h"
#include "ygNode.h"
	
extern "C" ygNode* construct_mayaAnimation(const char* name,bool master) { return new mayaAnimation(name,master); }

mayaAnimation::mayaAnimation(const char* name,bool master) : ygNode (name,master)
	{
	
	setClassName("mayaAnimation",true);
	p_ = new struct _mayaAnimationPrivateData;
	p_->active = false;
	p_->pfgroup=  new pfGroup;
	setPfNode(p_->pfgroup);
	p_->currentClip = 0;
	p_->firstFrame = true;
	p_->timeScale = 1.0;
	p_->netStartClip=0;
	p_->netActive=false;
	p_->autoStop=true;

	addNetKey("file",&p_->netClipFileName,YG_NET_STRING);
	addNetKey("start",&p_->netStartClip,YG_NET_INT);
	addNetKey("clr",&p_->netClearClips,YG_NET_INT);

	reliableKey("file");
	reliableKey("start");
	reliableKey("clr");
	
	debugFlag("verbose", &p_->debugVerbose);
	}

mayaAnimation::~mayaAnimation(void)
	{
	}

void mayaAnimation::reset(void)
	{
	p_->firstFrame = true;
	p_->active=false;
	p_->currentClip=0;
	
	clearClips ();
	ygNode::reset();
	}

void mayaAnimation::message(const ygMessage& msg)
	{

	ygString value;

	// load a clip
	
	if (msg == "file") {
		float offset=0;
		if (msg.args.size()>1)
			offset = msg.floatArg(1);

		loadFile(msg.args[0],offset);
		p_->netClipFileName=msg.args[0];
		netKeyChanged("file");
	}
	
	// start a clip
	else if (msg == "start" )
	{	
		if (msg.args.size()<1) 
			msg.error(name(),"(wrong number of arguments");
		else {
			int clipnumber = msg.intArg(0);
			startClip (clipnumber);
			
			p_->netStartClip=clipnumber;
			netKeyChanged("start");
		}
	}
	else if (msg == "stop" )
	{
		stopClip();

	}
	else if (msg == "reset")
	{
		clearClips ();
		netKeyChanged("clr");
	}
	else if (msg == "speed")
	{
		p_->timeScale=msg.floatArg(0);
		setTimeScale(p_->timeScale);
		
	}
	else if (msg == "postInfinity")
	{
		if (msg.args.size()>1)
			setPostInfinity(msg.intArg(0),msg.args[1]);
		else if (msg.args.size() == 1)
		{
			for (int i=0;i<p_->clips.size();i++)
				setPostInfinity(i,msg.args[0]);
		}
		
	}
	else if (msg == "preInfinity")
	{
		if (msg.args.size()>1)
			setPreInfinity(msg.intArg(0),msg.args[1]);
		else if (msg.args.size() == 1)
		{
			for (int i=0;i<p_->clips.size();i++)
				setPreInfinity(i,msg.args[0]);
		}
	}
	else if (msg == "loop")
	{
		ygString value;
			
		if (msg.args.size()>1) 
		{
			if (msg.boolArg(1))
				value = "cycle";
			else
				value = "constant";

			setPreInfinity(msg.intArg(0),value);
			setPostInfinity(msg.intArg(0),value);
		}
		else if (msg.args.size() == 1)
		{
			if (msg.boolArg(0))
				value = "cycle";
			else
				value = "constant";

			for (int i=0;i<p_->clips.size();i++)
			{
				setPreInfinity(i,value);
				setPostInfinity(i,value);
			}
		}
		else
		{
			for (int i=0;i<p_->clips.size();i++)
			{
				setPreInfinity(i,"cycle");
				setPostInfinity(i,"cycle");
			}
		}		
	}
	else if (msg == "oscillate")
	{
		if (msg.args.size()>1) 
		{
			if (msg.boolArg(1))
				value = "oscillate";
			else
				value = "constant";

			setPreInfinity(msg.intArg(0),value);
			setPostInfinity(msg.intArg(0),value);
		}
		else if (msg.args.size() == 1)
		{
			if (msg.boolArg(0))
				value = "oscillate";
			else
				value = "constant";

			for (int i=0;i<p_->clips.size();i++)
				setPreInfinity(i,value);
		}	
		else
		{
			for (int i=0;i<p_->clips.size();i++)
			{
				setPreInfinity(i,"oscillate");
				setPostInfinity(i,"oscillate");
			}
		}
			
	}
	
	else
		ygNode::message(msg);
	}

void mayaAnimation::app (void)
{
	
 	if (p_->active) {
 		if (p_->currentClip)
		{
			mayaAnimation::animatedClip *clip = p_->clips[p_->currentClip-1];
			
			if (!clip->initializationComplete)
				clip->attachNodes (p_->pfgroup);
			clip->evaluate(ygWorld::FrameTime);
			
			if (clip->beyondEnd())
			{
				eventOccurred("end");
				eventOccurred("back");
				if (p_->autoStop && clip->postInfinityIsConstant())
					stopClip();
			}
			
			if (clip->beyondBeginning())
			{
				eventOccurred("end");
				eventOccurred("front");
				if (p_->autoStop && clip->preInfinityIsConstant())
					stopClip();	
			}
			if (clip->loopOccurred())
			{
				eventOccurred("looped");
			}
			
		}
 	}
 
 ygNode::app();
}

int mayaAnimation::loadFile (const ygString& file,float offset)
{
	int err=0;
	if (p_->debugVerbose)
		fprintf (stderr,"loading file:%s\n",file.c_str());
		
	animatedClip * newclip = new animatedClip(p_->debugVerbose);
	p_->clips.push_back (newclip);
	
	err = newclip->loadFile(file);
	
	if (!err)
	{
		if (p_->debugVerbose)
			fprintf (stderr,"made new clip\n");
		newclip->initialize (p_->pfgroup);
		newclip->setOffset (offset);
	}
	return err;
}

void mayaAnimation::startClip (int clipnumber)
{
	ygString nodename;
	ygString channelname;

	// stop the previous clip
	/*
	if (p_->currentClip>0)
		p_->clips[p_->currentClip-1]->stop();
	*/
	if (clipnumber>0 && clipnumber<=p_->clips.size()) {
	
	p_->currentClip = clipnumber;
	p_->active = true;
	// start the new clip
	if (p_->debugVerbose)
		fprintf (stderr,"starting clip:%d\n",p_->currentClip);
	p_->clips[p_->currentClip-1]->setTimeScale(p_->timeScale);
	p_->clips[p_->currentClip-1]->start(nodename,channelname,ygWorld::FrameTime);
	
	}
}


void mayaAnimation::stopClip()
{
	if (p_->active && p_->currentClip>0)
	{
		p_->clips[p_->currentClip-1]->stop();
		p_->currentClip=0;
		p_->active=false;
	}
}

void mayaAnimation::setTimeScale(float s)
{
	if (p_->active)
	{
		if (p_->currentClip)
		{
			mayaAnimation::animatedClip *clip = p_->clips[p_->currentClip-1];
			clip->setTimeScale(s);
		}
	}
}
			
void mayaAnimation::clearClips ()
{
	p_->active = false;
	p_->currentClip=0;
	int i;
	for (i=0;i<p_->clips.size();i++)
		delete (p_->clips[i]);
	p_->clips.resize(0);
	p_->firstFrame=true;
}

void mayaAnimation::setPreInfinity(int c,const ygString& s)
{
	
	if (s=="constant")
	{
		p_->clips[c-1]->setPreInfinity(kInfinityConstant);
	}
	else if (s=="linear")
	{
		p_->clips[c-1]->setPreInfinity(kInfinityLinear);
	}
	else if (s=="cycle")
	{
		p_->clips[c-1]->setPreInfinity(kInfinityCycle);
	}
	else if (s=="cycleRelative")
	{
		p_->clips[c-1]->setPreInfinity(kInfinityCycleRelative);
	}
	else if (s=="oscillate")
	{
		p_->clips[c-1]->setPreInfinity(kInfinityOscillate);
	}
	else
	{
		p_->clips[c-1]->setPreInfinity(kInfinityConstant);
	}
}
void mayaAnimation::setPostInfinity(int c,const ygString& s)
{
	
	if (s=="constant")
	{
		p_->clips[c-1]->setPostInfinity(kInfinityConstant);
	}
	else if (s=="linear")
	{
		p_->clips[c-1]->setPostInfinity(kInfinityLinear);
	}
	else if (s=="cycle")
	{
		p_->clips[c-1]->setPostInfinity(kInfinityCycle);
	}
	else if (s=="cycleRelative")
	{
		p_->clips[c-1]->setPostInfinity(kInfinityCycleRelative);
	}
	else if (s=="oscillate")
	{
		p_->clips[c-1]->setPostInfinity(kInfinityOscillate);
	}
	else
	{
		p_->clips[c-1]->setPostInfinity(kInfinityConstant);
	}
}
void mayaAnimation::acceptNetKey(const ygString& key)
{
	if (key=="file")
	{
		loadFile(p_->netClipFileName,0);
	}
	else if (key=="start")
	{
		startClip(p_->netStartClip);
	}
	else if (key=="clr")
	{
		clearClips ();
	}
	else
		ygNode::acceptNetKey(key);
}
