/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/

#ifndef _CAVERNPLUS_CONDITION_C
#define _CAVERNPLUS_CONDITION_C

#include "CAVERNts_mutex_c.hxx"

#ifdef CAVERN_USE_GLOBUS_THREADS       
	typedef globus_cond_t CAVERN_COND_T;
#elif defined(CAVERN_USE_PTHREADS)
	typedef pthread_cond_t CAVERN_COND_T;
#elif defined(CAVERN_USE_WINDOWSTHREADS)
    class _CAVERN_COND_T 
    {
    public:
        _CAVERN_COND_T() {
            waiters = 0;
            wasBroadcast = 0;
            sema = CreateSemaphore(NULL, 0, 0xFFFFFFFF, NULL);
            waitersDone = CreateEvent(NULL, FALSE, FALSE, NULL);
            waitersLock = CreateMutex(NULL, FALSE, NULL);
        }

        virtual ~_CAVERN_COND_T() {
            if (waitersDone) CloseHandle(waitersDone);
            if (waitersLock) CloseHandle(waitersLock);
            if (sema) CloseHandle(sema);
        }

        long waiters;  // Number of waiting threads.
        HANDLE waitersLock; // Serialize access to the waiters count.

        // Queue up threads waiting for the condition to become signaled.
        HANDLE sema; 

        // An auto reset event used by the broadcast/signal thread to wait
        // for the waiting thread(s) to wake up and get a chance at the
        // semaphore.
        HANDLE waitersDone;

        // Keeps track of whether we were broadcasting or just signaling.
        size_t wasBroadcast;
    };

    typedef _CAVERN_COND_T * CAVERN_COND_T;
#else
	#error One of CAVERN_USE_GLOBUS_THREADS, CAVERN_USE_PTHREADS, or CAVERN_USE_WINDOWSTHREADS must be defined.
#endif

/**
Class for thread condition variables.
Condition variables are used in conjunction with
mutexes to provide a way for threads to wait until
a condition occurs.

An example of waiting on a signal is:

\begin{verbatim}

// Lock your mutex that is protecting someState.
myMutex->lock();

// Watch for your desired state to occur.
while(someState != reached) {

	// Wait for a signal.
	myCondition->wait(myMutex);

	.... got the condition and the lock so now continue ....

}

myMutex->unlock();

\end{verbatim}

An example of sending the signal is:

\begin{verbatim}

// Lock your mutex that is protecting someState.
myMutex->lock();

// Signal that the state has been reached.
if (someState == reached) myCondition->signal();

// Unlock your mutex so that the waiting thread can continue.
myMutex->unlock();

\end{verbatim}
*/

class CAVERNts_condition_c {
public:

	/** Wait on a condition to be signalled.
	    This function first releases the mutex and then
	    waits on the condition. When the condition
	    arises (ie it has been signaled) the mutex is
	    reaquired, and the function returns.
	    @return 0 if function successfully completes else non-zero
	    */
	int wait(CAVERNts_mutex_c *mutex);

	/** Signal that a condition has arisen. This wakes up
	    one thread that is suspended on this condition. If
	    no threads are suspended this call has no effect.

	    @return 0 if function successfully completes else non-zero


	    */
	int signal();

	/** Signal that a condition has arisen. This wakes up
	    ALL threads that are suspended on this condition. If
	    no threads are suspended this call has no effect.

	    @return 0 if function successfully completes else non-zero
	    */
	int broadcastSignal();

	/// Return condition variable.
	CAVERN_COND_T *getCondition();

	CAVERNts_condition_c();
	~CAVERNts_condition_c();

private:
	CAVERN_COND_T itsCV;
};


#endif

