/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/

/*
CAVERNnet_udpReflector_c
- setIncomingPort(int)
- setTimeOutTime(int)
- setForcedDestination(ip,port);
- newDataCB()
- deliver(data)
- setMulticast(ON or OFF);
- setMulticastPort(int)
- setMulticastAddress()

CAVERNnet_tcpReflector_c
- setIncomingPort(int)
- setTimeOutTime(int)
- setForcedDestination(ip,port);
- newDataCB()
- deliver(data)
*/

#include "CAVERNnet_udp_c.hxx"
#include "CAVERNnet_mcast_c.hxx"

#ifndef CAVERNNET_UDP_REFLECTOR
#define CAVERNNET_UDP_REFLECTOR

#define CAVERNNET_UDP_REFLECTOR_MAX_MSG_SIZE 8192
#define CAVERNNET_UDP_REFLECTOR_SEND_TO_ALL_UDP_CLIENTS -1
#define CAVERNNET_UDP_REFLECTOR_MAX_NUM_CLIENTS 1024
#define CAVERNNET_UDP_REFLECTOR_DEFAULT_TIMEOUT 60
#define CAVERNNET_UDP_REFLECTOR_IP_SIZE 256
#define CAVERNNET_UDP_REFLECTOR_MCAST_TTL 160
#define PERFDAEMON_ALIVE 1
#define PERFDAEMON_NOT_ALIVE 0

/** UDP and Multicast Reflector class.
Even though CAVERNsoft's key, link, and channel concept essentially acts as a data
reflector there are  instances where a generic packet reflector (to support legacy
applications) are needed. This class fulfills that need.

This class will accept incoming UDP packets and broadcast
them to all listeners. This class also accepts incoming multicast packets and broadcasts them to
all listeners (whether they are on UDP or Multicast). In that case
all incoming UDP packets will also be broadcasted to the Multicast group.

To add a listener to the reflector the remote client simply needs to send a packet to
the reflector. The reflector will use the packet's address as the return address
for future data. If a client does not send any data for a certain period  of time
that client is removed from the reflector's list of listeners.

*/
class CAVERNnet_udpReflector_c : public CAVERNnet_socketbase_c
{
public:
	CAVERNnet_udpReflector_c();
	~CAVERNnet_udpReflector_c();

	//@{
	/// Status Failed
	static const int FAILED/* = 0*/;

	/// Status Ok.
	static const int OK/* = 1*/;

    /// Skip the data distribution process. Used in user callback. See intercept().
        static const int SKIP_DISTRIBUTION/* = 2*/;
        
	//@}

	/** Initialize the reflector once all the parameters are set up.
	    If you need to change the settings at a later time you need to delete
	    this object and create a new one with the new settings.
	    @return FAILED if failed; OK if success.
	 */
	int init();

	/** Set the incoming port for the reflector to listen on. Default is set at 6000 */
	void setIncomingPort(unsigned short port = 6000);

	/** Set the time out time when connected clients who have not sent any data for
	a while will be removed. Default is set at 60 seconds.
	*/
	void setTimeOutTime(int secs);

	/// Set a single ip address and port number to which all packets will be sent.
	void setForcedDestination(char *ipAddr, unsigned short port);

	/** Set up to also listen on a multicast address and port.
	    This means that any incoming packets on the UDP connection will get broadcasted
	    via this reflector to the multicast channel too.
	*/
	void setMulticastAddress(char *ipAddr,unsigned short port);

	/** Set multicast TTL to something other than the default at 160 */
	void setMulticastTTL(int ttl);

	/** Call this within a while loop to let the reflector continuously
	    do its processing.
	    @return FAILED if error; OK if success.
	*/
	int process();

    /** Intercept incoming messages and call a user-defined callback function.
    If you want you can also alter the buffer completely so that the reflector will reflect an entirely different message. You can do this by changing the contents of the buffer or  by replacing the buffer entirely by allocating memory for a new buffer and stuffing it with your own data. If you choose to allocate a totally new buffer you must remember  to deallocate memory for the original buffer before substituting it with yours.                                              If after your callback function exits you do not wish the reflector to forward  the contents of the buffer, return with CAVERNnet_udpReflector_c::SKIP_DISTRIBUTION. Otherwise just return CAVERNnet_udpReflector_c::OK.
    
    Note also that the callback function will also be given a pointer to a CAVERNnet_udp_c object that can then be used to send data directly to the client that originally sent the message.
                                                                                    */
     void intercept(int (*callback) (CAVERNnet_udp_c *client, char** buffer, int *bufferSize,  void *userData), void* userData);
  
                                                                                     /**Enable instrumentation - this is for turning on the performance monitoring for the UDPsocket used -instrumentation for the udp sockets are turned off by default - and it should be explicitly enabled for latency, jitter and related statistics
     */
    void enableInstrumentation();
    
    /**Disable instrumentation - this is for turning off the performance monitoring for the UDPsocket used*/ 

    void disableInstrumentation();

    //Functions added for performance monitoring interface

    /**
       Displays the resultant statistics instantaneously in the netlogger format - this should be typically done after a read/write is done  a network.

       Also, it should be noted that a showStats call should be made at the end of atleast one send and receive for two-way information (the same applies for logStats and sendStats)

       	@param streamInfo
        A label describing the stream that is being monitored.

        @param comment
        A comment on the event that marks the time at which the stream is being monitored
     */
    void showStats(char* streamInfo, char* comment);
    
    /**
       This logs performance statistics in a file. The user opens a file and passes the file pointer with this function and results of monitoring are written into the logfile.
      
       @param streamInfo
        A label describing the stream that is being monitored.

        @param comment
        A comment on the event that marks the time at which the stream is being monitored
        
        @param filePtr
        File pointer to the file in which the results of monitoring are to be stored

        @return
        Either CAVERNnet_perfMonitor_c::OK or CAVERNnet_perfMonitor_c::FAILED
     */

    int logStats(char* streamInfo, char* comment, FILE* filePtr);

    /**
       Sends the performance statistics to a remote perfdaemon -for further analysis of the monitored data - the initSendStats API should be called first, before calling a sendStats (In order to connect to the perfdaemon initially) 


       @param streamInfo
        A label describing the stream that is being monitored.

        @param comment
        A comment on the event that marks the time at which the stream is being monitored
 
        @return
        Either CAVERNnet_perfMonitor_c::OK or CAVERNnet_perfMonitor_c::FAILED
       
     */

    int sendStats(char* streamInfo, char* comment);

    /**
       Initialize sendStats - provide the IP of the perfDaemon and an optional port number to connect to. This should be done initially  before using the sendStats API.

       @param monitorClientIP
       IP address of the perfDameon to connect to
       
       @param port
       Port number at which the perfDaemon is running -this is optional. The default port number for perfDaemon is 9500 -so a different port number has to be specified if the perfDaemon is running ona different port.

       @return
       Either CAVERNnet_perfMonitor_c::OK or CAVERNnet_perfMonitor_c::FAILED
     */

    int initSendStats(char* monitorClientIP, int port = PERF_DAEMON_DEFAULT_PORT);
    
    /**
       Properly delete the perfDaemonClient after sendStats is done
     */

    void exitSendStats();

private:
	double *itsAliveClients;
	CAVERNnet_udp_c *itsForcedUDP, *itsMainUDPSocket;
	CAVERNnet_mcast_c *itsMcastSocket;
	unsigned short itsUDPPort;
	int itsTimeOutTime;
	int itsMcastTTL;
	double itsStartTime;
	char itsForcedUDPIP[CAVERNNET_UDP_REFLECTOR_IP_SIZE];
	char itsMcastIP[CAVERNNET_UDP_REFLECTOR_IP_SIZE];
	unsigned short itsMcastPort;
	unsigned short itsForcedUDPPort;
	CAVERNnet_udp_c **itsBunchOfClients;
	int itsNumberOfClients;
	int doMcast, doForcedUDP;
	double itsLastTime;
	char* inBuffer; 
	int itsInit;
	int initClientList();
	int addClient(CAVERNnet_udp_c*);
	void replyToClients(char *mesg, int size, int exceptClientIndex);
	void purgeOldClients();
	int  checkClient(CAVERNnet_udp_c *mainUDPSocket);
    
    int (*interceptCallback) (CAVERNnet_udp_c *newClient, char** buffer, int* bufsize, void *userData);
    void *interceptUserData;
   
    char* perfDaemonIP;     //IP of the perf-daemon - used for initSendStats
    int perfDaemonPort;     //PerfDaemon port
    int perfDaemonStatus;   //Status that indicates if the perfdaemon is running

};
#endif
