/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/
/*
This is a C++ class to support TCP/IP sockets.
The 2 main classes are CAVERNnet_tcpClient_c and CAVERNnet_tcpServer_c.

If you are writing a server you need to instantiate a tcpServer object.
Likewise if you are writing a client you need to instantiate a tcpClient
object.

*/

#ifndef _CAVERNPLUS_TCP_C
#define _CAVERNPLUS_TCP_C

#ifdef WIN32
#include <errno.h>
#elif defined(__sgi)
extern int errno;
#endif

#include "CAVERNnet_socketbase_c.hxx"
#include "CAVERNnet_perfMonitor_c.hxx"
//#include "CAVERNnet_perfDaemon_c.hxx"

class CAVERNnet_perfMonitor_c;
class CAVERNnet_tcpServer_c;
//class CAVERNnet_perfDaemon_c;

/** TCP Class for creating client connections. This class bypasses CAVERN's standard method for establishing connections and provides the user with direct control of TCP connections.

*/
class CAVERNnet_tcpClient_c : public CAVERNnet_socketbase_c {

	// Allow CAVERNnet_tcpServer_c objects to call MakeNonBlocking.
	friend class CAVERNnet_tcpServer_c;

protected:
	/* Not to be called by user */
	void makeNonBlocking();

	struct sockaddr_in clientInfo;
	int clientSockFd;
	int clientStatus;
	int timeOutStatus;
	int timeOutPeriod;
    CAVERNnet_perfMonitor_c pmonitor;
public:

	/// Blocking parameter values for Read/Write method.
	//@{
	/// Make the connection blocking but time out eventually.
	static const int BLOCKING/* = 1*/;

	/// Make the connection non blocking.
	static const int NON_BLOCKING/* = 2*/;

	// Make the conncection blocking and never time out.
	//static const int NEVER_TIME_OUT_BLOCKING/* = 3*/;

	//@}

	/// Time out values.
	//@{
	/// Used to set time out period to never time out.
	static const int NO_TIME_OUT/* = -1*/;

	/// Default time out is no time out.
	static const int DEFAULT_TIME_OUT/* = NO_TIME_OUT*/; 
	//@}

	/// Return values from Read/Write methods.
	//@{
	/// Socket is not open.
	static const int SOCKET_NOT_OPEN/* = -1*/;

	/// Connection you attempted to read/write is terminated.
	static const int CONNECTION_TERMINATED/* = -2*/;

	/// You attempted to read with non blocking and there was not data.
	static const int NON_BLOCKING_HAS_NO_DATA/* = -3*/;

	/// The read/write timed out.
	static const int TIMED_OUT/* = -4*/;

	/// Read/Write succeeded.
	static const int OK/* = 1*/;

	/// You attempted to write with non blocking and socket was not ready.
	static const int NON_BLOCKING_NOT_READY_TO_WRITE/* = -6*/;

	/// Socket was not ready.
	static const int NOT_READY/* = -7*/;

	/// Socket ready to read.
	static const int READY_TO_READ/* = -8*/;

	/// Socket ready to write.
	static const int READY_TO_WRITE/* = -9*/;

	/// Socket ready to read and write.
	static const int READY_TO_READ_AND_WRITE/* = -10*/;

    /// Change the socket buffer size before Read().
    static const int READ_BUFFER_SIZE/* = -11*/;

    /// Change the socket buffer size before Write().
    static const int WRITE_BUFFER_SIZE/* = -12*/;
	//@}

	
	/** Set timeout period when performing read() or write() operations.
	    @param t Time out period in seconds. Set to CAVERNnet_tcpClient_c::NO_TIME_OUT to set no time out period.
	*/
	void setTimeOut(int t) {timeOutPeriod = t;}

	/// Get timeout period
	int getTimeOut() {return timeOutPeriod;}

	/** Set client info. For internal use.
	    @param info client information
	*/
	void setClientInfo(struct sockaddr_in* info) { clientInfo = *info; }

	/** Set client socket. For internal use.
	    @param sockfd client socket
	*/
	void setClientSockFd(int sockfd) { clientSockFd = sockfd; }

	CAVERNnet_tcpClient_c();
	~CAVERNnet_tcpClient_c() { if (clientSockFd) close(); }

	/// Accessor functions
	//@{
	/** Get the IP address of remote connection.
	If you are a client this returns the ip of the destination server.
	If you are a server this returns the ip of the destination client.
	*/
	unsigned int getRemoteIP();

	/** Get the IP address of remote connection.
	If you are a client this returns the ip of the destination server.
	If you are a server this returns the ip of the destination client.
	*/
	void getRemoteIP(char* name);

	/// Get IP address of self.
	unsigned int getSelfIP();

	/// Get IP address of self.
	void getSelfIP(char *name);

	/// Get port of self.
	int  getSelfPort();

	/// Get port of client.
	int  getRemotePort();

	/// Get socket id
	int getSocketId();
	//@}

	/**@name Read and Write calls.*/
	//@{
	/**
	  @param nbytes

	  Fill nbytes with num bytes you want to read.
	  nbytes will return with number of bytes successfully read.

	  @param blockingtype

	  BLOCKING means that it will block waiting for data or until
	  the timeout period expires.  Change the timeout period by
	  calling setTimeOut().

	  NON_BLOCKING means that if there is no data to be read this call will
	  return immediately. If it does get some data it will keep reading
	  until the specified number of bytes is received, or if it timesout, or if
	  the connection is broken. In the latter two cases the number of bytes
	  it was able to read is returned.

	  In either the BLOCKING or NON_BLOCKING case you can set the
	  time out period to never time out by setting the timeout
	  time to: NO_TIME_OUT.  

	  @return SOCKET_NOT_OPEN, NON_BLOCKING_HAS_NO_DATA, TIMED_OUT, CONNECTION_TERMINATED, OK, errno
	  */
	int read(char *ptr, int *nbytes, int blockingType);

	/** Allows you to change the parameters to the setsockopt() options.Currently this member allows you to change socket buffer size.

	    @param option 
	    Specify CAVERNnet_tcpClient_c::READ_BUFFER_SIZE if you want to change the size of the receive buffer.
	    Specify CAVERNnet_tcpClient_c::WRITE_BUFFER_SIZE if you want to change the size of the send buffer			

	    @param buffersize
	    Specify the size.
	*/	
	void setSockOptions(int option, int buffersize);	


	/** Determines if a socket has data available to read.
	    @return Either: CAVERNnet_tcpClient_c::NOT_READY
	    or CAVERNnet_tcpClient_c::READY_TO_READ
	*/
	int isReadyToRead();

	/** Determines if a socket is ready to write.
	    @return Either: CAVERNnet_tcpClient_c::NOT_READY
	    or CAVERNnet_tcpClient_c::READY_TO_WRITE
	*/
	int isReadyToWrite();

	/** Determines if a socket is ready to write or read or both.
	    @return Either: CAVERNnet_tcpClient_c::NOT_READY
	    or CAVERNnet_tcpClient_c::READY_TO_WRITE
	    or CAVERNnet_tcpClient_c::READY_TO_READ
	    or CAVERNnet_tcpClient_c::READY_TO_READ_AND_WRITE
	*/
	int isReady();

	/** Write data to socket.
	@return SOCKET_NOT_OPEN, NON_BLOCKING_NOT_READY_TO_WRITE, TIMED_OUT, CONNECTION_TERMINATED, OK
	*/
	int write(char *ptr, int  *nbytes, int blockingType);

	//@}

	/// Show status of connection in English.
	void showStatus(int status, int nbytes);

	/// Close the current client connection.
	void close();

	/** Connect to a server.
	    @return A negative number(that is returned by a socket sonnect() call if failed. Else returns client's socket file descriptor. (note: the return value in the previous versions was a zero in case of an error - this has been changed to a negative value from version 1.2)
	*/
	int connectToServer(char *ip, int port);


    //Functions added for performance monitoring interface
    /**
       Displays the resultant statistics instantaneously in the netlogger format - this should be typically done after a read/write is done over a network.
       
       Also, it should be noted that a showStats call should be made at the end of atleast one send and receive for two-way information (the same applies for logStats and sendStats)
       
       @param streamInfo
       A label describing the stream that is being monitored.
       
       @param comment
       A comment on the event that marks the time at which the stream is being monitored
    */

    void showStats(char* streamInfo, char* comment);
    
    /**
       This logs performance statistics in a file. The user opens a file and passes the file pointer with this function and results of monitoring are written into the logfile.
      
       @param streamInfo
        A label describing the stream that is being monitored.

        @param comment
        A comment on the event that marks the time at which the stream is being monitored
        
        @param filePtr
        File pointer to the file in which the results of monitoring are to be stored

        @return
        Either CAVERNnet_perfMonitor_c::OK or CAVERNnet_perfMonitor_c::FAILED
     */


    int logStats(char* streamInfo, char* comment, FILE* filePtr);

    /**
       Sends the performance statistics to a remote perfdaemon -for further analysis of the monitored data - the initSendStats API should be called first, before calling a sendStats (In order to connect to the perfdaemon initially) 

       @param streamInfo
       A label describing the stream that is being monitored.
       
       @param comment
       A comment on the event that marks the time at which the stream is being monitored
       
       @return
       Either CAVERNnet_perfMonitor_c::OK or CAVERNnet_perfMonitor_c::FAILED
       
     */

    int sendStats(char* streamInfo, char* comment);

    /**
       Initialize sendStats - provide the IP of the perfDaemon and an optional port number to connect to. This should be done initially  before using the sendStats API.
       
       @param monitorClientIP
       IP address of the perfDameon to connect to
       
       @param port
       Port number at which the perfDaemon is running -this is optional. The default port number for perfDaemon is 9500 -so a different port number has to be specified if the perfDaemon is running ona different port.
       
       @return
       Either CAVERNnet_perfMonitor_c::OK or CAVERNnet_perfMonitor_c::FAILED
    */

    int initSendStats(char* monitorClientIP, int port = PERF_DAEMON_DEFAULT_PORT);
    
    /**
       Properly delete the perFDaemon Client after sendStats is done
     */

    void exitSendStats();

    //Set the instantaneous latency 
    void setInstantaneousLatency(double latency);


};

/** TCP Class for creating servers. This class bypasses CAVERN's standard method for establishing connections and provides the user with direct control of TCP connections.
*/

class CAVERNnet_tcpServer_c : public CAVERNnet_socketbase_c {


protected:
	int serverPort;

	int timeOutPeriod;
	int sockfd;
	struct sockaddr_in  serverInfo;

public:

	//@{
	/// Status ok.
	static const int OK/* = 1*/;

	/// Status failed.
	static const int FAILED/* = 0*/;
	//@}

	//	static int CAVERNnet_tcpServer_c::NO_TIME_OUT;

	/** Set timeout period when performing read() or write() operations.
	    @param t Time out period in seconds. Set to CAVERNnet_tcpClient_c::NO_TIME_OUT to set no time out period.
	*/
	void setTimeOut(int t) {timeOutPeriod = t;}

	/// Get timeout period.
	int getTimeOut() {return timeOutPeriod;}

	CAVERNnet_tcpServer_c();
	~CAVERNnet_tcpServer_c() { if (sockfd) close(); }

	/// Get server's IP.
	unsigned int getSelfIP();

	/// Get server's IP
	void getSelfIP(char* name);

	/// Get server's port.
	int getSelfPort() {
		return ntohs(serverInfo.sin_port);
	}

	/** Open the server on a port.
	 Typically after this call you sit in a loop and call
	 checkForNewConnections to wait for incoming connections.
	 @return FAILED if failed, OK if success
	 */
	int init(int port);

	/// Close the server port.
	void close();

	/** Check to see if there is a request from clients for connection.
	 * If yes then return a CAVERNnet_tcpClient_c object which you can then use
	 * to talk to the client.
	 * @see CAVERNnet_tcpClient_c class.
	 */
	CAVERNnet_tcpClient_c *checkForNewConnections();

	/*
        This function checks for a new connection and is the same as the   			previous function except that it blocks for a user specified timeout and 	returns
    	*/
	CAVERNnet_tcpClient_c *checkForNewConnections(int blockingTime);
};

#endif
