/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/

#ifndef _CAVERNPLUS_SOCKETBASE_C
#define _CAVERNPLUS_SOCKETBASE_C

#ifdef CAVERN_USE_GLOBUS_THREADS
#include "globus_common.h"
#endif

#ifdef CAVERN_USE_PTHREADS
#include <pthread.h>
#endif

#include <stdio.h>
#ifndef _WIN32_WCE
   #include <fcntl.h>
#endif
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
//#include <fstream.h>

#ifdef WIN32
#include <windows.h>
#include <winsock.h>
#include <sys\timeb.h>
#else
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/uio.h>
#include <arpa/inet.h>
#include <sys/errno.h>
#include <signal.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/time.h>
#endif

// added for the socklen_t fix for Linux and FreeBSD
#ifdef __sgi
typedef int socklen_t;
#endif


//These are the number of packets for which all calculations are done -
//this is to be changed as desired

#define LATENCY_BUF_SIZE 100
#define BANDWIDTH_BUF_SIZE 100

extern int errno;

/** A base level socket class subclassed by CAVERNnet_tcp_c, CAVERNnet_udp_c, CAVERNnet_mcast_c, CAVERNnet_parallelTcp_c,  and CAVERNnet_perfMonitor_c. The class itself does no networking. It provides some common member functions that are useful for network coding. In general one does not create an instance of this type.

    This class also offers information about the bandwidth, latency, Inter-Message_delay, Jitter and Burstiness values associated with the individual socket connections. Latency and Jitter calculations are offered only if the individual subclasses have implemented and enabled it. Otherwise values will be zero.
 These calculations are used by the CAVERNnet_perfMonitor_c class
*/

class CAVERNnet_socketbase_c {
public:

	///Socketbase class return values
	//@{
	/// Operation went ok
	static const int OK/* = 1*/;
	/// Operation failed
	static const int FAILED/* = 0*/;
	/// Operation timed out
	static const int TIMED_OUT/* = 2 */;
	//@}

	CAVERNnet_socketbase_c();

	/// Convert a hostname to an ip address of xxx.xxx.xxx.xxx
	static int hostnameToIP(char *hostname, char* hostIP);

	/// Convert ip address to a string of xxx.xxx.xxx.xxx
	static void ipNumToString(unsigned int ip, char *ipstring);

	/// Get time in secs with microsecond accuracy
	static double getTimeInSecs();

	/// Get the full host name of a local client
	static void getHostName(char *name, int len);

	/// If hostname is not fully expanded (ie does not include its domain) then expand it.
	static void expandHostName(char *src, char *newname, int len);

	/// Get the total data sent by a connection.
	unsigned int getTotalDataSent();

	/// Get total data received by a connection.
	unsigned int getTotalDataRead();

	/// Get total instantaneous incoming bandwidth by a connection.
	double getInstantReceiveBandwidth();

	/// Get total instantaneous outgoing bandwidth by a connection.
	double getInstantSendBandwidth();

	/// Get average incoming bandwidth by a connection.
	double getAverageReceiveBandwidth();

	/// Get average outgoing bandwidth by a connection.
	double getAverageSendBandwidth();

    ///Set the instantaneous latency and also calculate the STALatency and Jitter values automatically
	void setInstantLatency(double lat);

	/// Retrieve most recent incoming packet's latency.
	double getInstantLatency();

	/// Retrieve average incoming latency.
	double getAverageLatency();

	/// Retrieve max latency so far.
	double getMaxLatency();

	/// Retrieve min latency so far.
	double getMinLatency();

	/// Record amount of data sent.
	void incrementDataSent(unsigned long size);

    /// Record amount of data read - and also compute instantaneous, average bandwidths along with the inter message delay values
	void incrementDataRead(unsigned long size);

	/// Get instantaneous inter-message delay (jitter)
	double getInstantInterMesgDelay();

	/// Get max inter-message delay (jitter)
	double getMaxInterMesgDelay();

	/// Get min inter-message delay (jitter)
	double getMinInterMesgDelay();

	/// Get average inter-message delay (jitter)
	double getAverageInterMesgDelay();

   ///Get the Short Term Average Bandwidth -over a fixed number of packets (defined by BANDWIDTH_BUF_SIZE)
    double getSTABandwidth();

    ///Get the Short Term Average Latency -over a number of packets (defined by LATENCY_BUF_SIZE)
    double getSTALatency();

    ///get the burstinesss
    double getBurstiness();   

    ///Get the Jitter 
    double getJitter();

    ///Return the total number of packets read
    unsigned long getPacketsRead();

    ///Return the total number of packets sent
    unsigned long getPacketsSent();

    ///Returns the socket id
    virtual int getSocketId();

    ///Returns the status of the sockets being polled
    int selectSock(void *ptr, int *status, int numSocks, 
struct timeval *timeout);  


protected:
	double instLat, latTotal, minLat, maxLat, avgLat;
	unsigned long latCount,sentCount, readCount;
	double instDelay, totalDelay, minDelay, maxDelay, avgDelay, prevLocalTime;
	double currentTime, previousSentTime, previousReadTime,initialSentTime, initialReadTime;
	unsigned long prevTotalDataRead, prevTotalDataSent, totalDataRead, totalDataSent;
	double instReadBandwidth, instSendBandwidth, avgReadBandwidth, avgSendBandwidth;

	//STA - Short Term Average -calculated for  a specified number of packets (calcualted over a number of packets defined by LATENCY_BUF_SIZE)

    //Jitter is the average of the deviations of the instantaneous latencies from the STALatency

    //Burstiness(for bandwidths) is calculated the same way as Jitter is calculated for latencies

    double STALatency, STABandwidth, jitter, burstiness;

    //The latencies and bandwidth are stored in a buffer of defined size and the short term sums -(STlatencySum and STBandwidthSum) are calculated
    double latencyBuffer[LATENCY_BUF_SIZE];
    double bandwidthBuffer[BANDWIDTH_BUF_SIZE];  
    double STlatencySum, STbandwidthSum;

    //to simplify Short Term Average calculations
    int tempLatCounter, tempBWCounter;

    ///Calculate the instantaneous outgoing bandwidth 
	void computeInstantSendBandwidth();

    ///Calculate the instantaneous incoming bandwidth 
    void computeInstantReceiveBandwidth();

    ///Calculate the average incoming bandwidth 
	void computeAverageReceiveBandwidth();

    ///Calculate the average outgoing bandwidth 
    void computeAverageSendBandwidth();

    ///Calculate the jitter over a specified number of packets 
    void calculateJitter(double lat);
    
    ///Calculate the burstiness over a specified number of packets 
    void calculateBurstiness(double bw);


};


#endif
