/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/

#ifndef _CAVERNPLUS_PARALLEL_TCP_C
#define _CAVERNPLUS_PARALLEL_TCP_C

#include "CAVERNnet_socketbase_c.hxx"
#include "CAVERNnet_datapack_c.hxx"

#ifdef WIN32
#include <errno.h>
#elif defined(__sgi)
extern int errno;
#endif

/**
Parallel socket class. This enables us to transfer a stream of data over 
multiple sockets simultaneously.

Why is this faster than a single socket connection? The TCP protocol requires 
acknowledgement from the other end everytime it sends a packet, thus it waits 
for someting on every send.  With parallel sockets, while one socket is waiting, 
another socket connection can send data.  We therefore eliminate idle time.

If you are writing a server you need to instantiate a CAVERNnet_parallelTcpServer_c object.
Likewise if you are writing a client you need to instantiate a CAVERNnet_parallelTcpClient_c
object.

Server setup:  First, instantiate the CAVERNnet_parallelTcpServer_c class. Then call 
this object's init() method with the desired port number as parameter. 
After this, we call CAVERNnet_parallelTcpServer_c::checkForNewConnections(PortNumber).  
This is a blocking call that waits for the client to make a 
CAVERNnet_parallelTcpClient_c::connectToServer() 
call. checkForNewConnections() returns a pointer to a CAVERNnet_parallelTcpClient_c object,
which you can use to communicate with the client. Note that you do not explicitly
instantiate a CAVERNnet_parallelTcpClient_c object when you are using the server class.
You can just declare a pointer to the client object and assign to this variable the return
value of CAVERNnet_parallelTcpServer_c::checkForNewConnections function.

Client setup:  We instantiate the CAVERNnet_parallelTcpClient_c class and call the 
CAVERNnet_parallelTcpClient_c::connectToServer(MachineName,PortNumber,NumberOfSockets) method. 
We then use this object to do a write and read.

@author: Stuart Bailey, cavern@evl 

@version: 2/22/2000
*/

class CAVERNnet_parallelTcpClient_c : public CAVERNnet_socketbase_c
{ 
public:
	CAVERNnet_parallelTcpClient_c();
    ~CAVERNnet_parallelTcpClient_c();

        ///Parallel socket class return values
        //@{
		/// Operation successful
		static const int OK	   /* = 1*/;
	        /// Operation failed
        	static const int FAILED/* = 0*/;
        //@}
	/**
	After returning from checkForNewConnections() or after calling connectToServer(), 
	you can now call write() to send Length bytes of data starting at Buffer

	@return
	Either CAVERNnet_parallelTcpClient_c::OK or CAVERNnet_parallelTcpClient_c::FAILED

	@param Buffer
	A pointer to the buffer to be sent.

	@param Length
	Number of bytes to be sent.
	*/
    int write( char* buffer, int* length );

	/**
	After returning from checkForNewConnections() or after calling connectToServer(), 
	you can now call write() to send Length bytes of data starting at Buffer
	
	@return
	Either CAVERNnet_parallelTcpClient_c::OK or CAVERNnet_parallelTcpClient_c::FAILED

	@param Buffer
	A pointer to the buffer that will store the received data.

	@param Length
	Number of bytes to be received.
	*/
    int read( char* buffer, int* length );


	/**
	Close down the multiple socket handles
	*/
    void close();

	/**
	We instantiate a CAVERNnet_parallelTcpClient_c object and calls this 
	connectToServer() method to connect to machine named "RemoteName" on port Port.  
	We specify the number of simultaneous socket connections in 
	the Size parameter. After this call is successfully called, you can then
	use read and write methods.

	@param RemoteName
	A character string, name of the machine where the remote server is.

	@param Port
	Port number which the remote server is listening to.
	
	@param Size
	The number of simultaneous sockets we are to use.

	@return 
	Either CAVERNnet_parallelTcpClient_c::OK or CAVERNnet_parallelTcpClient_c::FAILED

	*/	
	int connectToServer(char *RemoteName, int RemotePort, int Size);

	/// Accessor functions
	//@{
	/** Get the IP address of remote connection.
	If you are a client this returns the ip of the destination server.
	If you are a server this returns the ip of the destination client.
	*/
	unsigned int getRemoteIP();

	/** Get the IP address of remote connection.
	If you are a client this returns the ip of the destination server.
	If you are a server this returns the ip of the destination client.
	*/
	void getRemoteIP(char* name);

	/// Get IP address of self.
	unsigned int getSelfIP();

	/// Get IP address of self.
	void getSelfIP(char *name);

	/// Get port of self.
	int  getSelfPort();

	/// Get port of client.
	int  getRemotePort();
	//@}

public:
	/* These functions are provided to be used in CAVERNnet_parallelTcpServer_c only. */
	void setSockets(int* Sockets, int NumSockets);/* { 
        sockets = Sockets;
        numSockets = NumSockets; 
        for (int i = 0; i < NumSockets; i++) {
            if (sockets[i] > maxSockDesc) maxSockDesc = sockets[i];
        }
        pcReadPtr = new char *[NumSockets];
        pcWritePtr = new char *[NumSockets];
        piBytesLeft = new int[NumSockets];
    }*/

protected:
	/* Not to be called by user */
	void makeNonBlocking(int Sockfd);
	int connectToClient(char* RemoteName, unsigned short RemotePort, 
						int& SocketInfo);

	char** pcReadPtr;
	char** pcWritePtr;
	int* piBytesLeft;
	int* sockets;    
	fd_set fdSet;
    int maxSockDesc;
    int numSockets;
    struct sockaddr_in remoteAddr;
    struct sockaddr_in localAddr;   
};

/**
If you are writing a server you need to instantiate a CAVERNnet_parallelTcpServer_c object.
Likewise if you are writing a client you need to instantiate a CAVERNnet_parallelTcpClient_c
object.

Server setup:  First, instantiate the CAVERNnet_parallelTcpServer_c class. Then call 
this object's init() method with the desired port number as parameter. 
After this, we call CAVERNnet_parallelTcpServer_c::checkForNewConnections(PortNumber).  
This is a blocking call that waits for the client to make a 
CAVERNnet_parallelTcpClient_c::connectToServer() 
call. checkForNewConnections() returns a pointer to a CAVERNnet_parallelTcpClient_c object,
which you can use to communicate with the client. Note that you do not explicitly
instantiate a CAVERNnet_parallelTcpClient_c object when you are using the server class.
You can just declare a pointer to the client object and assign to this variable the return
value of CAVERNnet_parallelTcpServer_c::checkForNewConnections function.

Client setup:  We instantiate the CAVERNnet_parallelTcpClient_c class and call the 
CAVERNnet_parallelTcpClient_c::connectToServer(MachineName,PortNumber,NumberOfSockets) method. 
We then use this object to do a write and read.

*/
class CAVERNnet_parallelTcpServer_c : public CAVERNnet_socketbase_c
{
public:
	//@{
	/// Status ok.
	static const int OK/* = 1*/;

	/// Status failed.
	static const int FAILED/* = 0*/;
	//@}


	CAVERNnet_parallelTcpServer_c() {};
	~CAVERNnet_parallelTcpServer_c() 
	{ 
#ifdef WIN32
		if (sockfd > 0) closesocket(sockfd); 
#else
		if (sockfd > 0) ::close(sockfd);
#endif
	}

	/** Open the server on a port.
	 Typically after this call you sit in a loop and call
	 checkForNewConnections to wait for incoming connections.
	 @return FAILED if failed, OK if success
	 */
	int init(int port);

	/// Close the server port.
	void close() { 
#ifdef WIN32
		closesocket(sockfd);
#else
		::close(sockfd);
#endif
		sockfd = 0;
	}

	/** Check to see if there is a request from clients for connection.
	 * If yes then return a CAVERNnet_parallelTcpClient_c object which you can then use
	 * to talk to the client.
	 * @see CAVERNnet_parallelTcpClient_c class.
	 */
	CAVERNnet_parallelTcpClient_c *checkForNewConnections();

	/// Get IP address of self.
	unsigned int getSelfIP();

	/// Get port of self.
	int getSelfPort() { return ntohs(serverInfo.sin_port); };

protected:
	void makeNonBlocking(int Sockfd);
	int connectToClient(unsigned long RemoteAddr, unsigned short RemotePort, 
						int& SocketInfo);

	struct sockaddr_in  serverInfo;
	int sockfd;
};

#endif
