/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/
 
 //  Based almost entirely on the work of Russel Winder

#ifndef _CAVERNmisc_observer_c
#define _CAVERNmisc_observer_c

#ifdef WIN32
#include <list>
#ifndef _WIN32_WCE
using namespace std;
#endif
#else
#include <list.h>
#endif



class CAVERNmisc_subject_c ;

/**
Observer class. This is used in conjunction with a CAVERNmisc_subject_c class.  

What is it for?  The subject-observer design pattern is an object oriented alternative to callback functions.  
We can have multiple observers "watch" a single subject.  An observer begins to watch a subject when the subject makes an "attach(&observer)" call.  Everytime the subject calls its notify() method, all observers that are watching it will react by calling their update() method.

We do not instantiate the CAVERNmisc_subject_c and the CAVERNmisc_observer_c.  Instead, the API user creates classes that inherit the properties of these two superclasses.  The class that inherits the observer class properties should implement the update() method.  The update method is a pure virtual function.  Not implementing it would result in a compiler error.

@author: cavern@evl	

@version: 12/1/99

*/
class CAVERNmisc_observer_c
{
 public:
  virtual ~CAVERNmisc_observer_c() {} ;  


  /**
	This method is executed whenever a subject this observer is "watching" calls its notify() method.  You can think of this as the equivalent of a callback function.  This is a pure virtual function that should be implemented by classes that inherit from the observer.  
	

	@param CAVERnmisc_subject_c 
	This is a pointer to the subject that woke us up.  We can then access its public member variables and even execute its public methods.
  */
  virtual void update(CAVERNmisc_subject_c * ) = 0 ;
 protected:
  CAVERNmisc_observer_c() {}

};

/**
Subject class. This is used in conjunction with a CAVERNmisc_observer_c class.  

What is it for?  The subject-observer design pattern is an object oriented alternative to callback functions.  
We can have multiple observers "watch" a single subject.  An observer begins to watch a subject when the subject makes an "attach(&observer)" call.  Everytime the subject calls its notify() method, all observers that are watching it will react by calling their update() method.

We do not instantiate the CAVERNmisc_subject_c and the CAVERNmisc_observer_c.  Instead, the API user creates classes that inherit the properties of these two superclasses.  The class that inherits the observer class properties should implement the update() method.  The update method is a pure virtual function.  Not implementing it would result in a compiler error.

@author: cavern@evl	

@version: 12/1/99

*/
class CAVERNmisc_subject_c
{
 public:
	virtual ~CAVERNmisc_subject_c() {};

  /**
	Subject calls attach(&observer) to register the observer.  After this, everytime the subject calls its notify() method, the registered observer reacts accordingly by calling its update() method.

	@param CAVERNmisc_observer_c 
	A pointer to the observer we want to register
  */
  virtual void attach(CAVERNmisc_observer_c *) ;


  /**
	Subject calls detach(&observer) to DEregister the observer.

	@param CAVERNmisc_observer_c 
	A pointer to the observer we want to DEregister
  */
  virtual void detach(CAVERNmisc_observer_c *) ;


  /**
	Subject calls notify() to tell all watching observers to react accoringly.  (Each watching observer will call its update method.)

  */
  virtual void notify() ;
 private:
  list<CAVERNmisc_observer_c *> m_ListOfObservers ;
};


inline void CAVERNmisc_subject_c::attach(CAVERNmisc_observer_c *o)
{
  m_ListOfObservers.push_back(o) ;
}

inline void CAVERNmisc_subject_c::detach(CAVERNmisc_observer_c *o)
{
  m_ListOfObservers.remove(o) ;
}

inline void CAVERNmisc_subject_c::notify() 
{
	
  for (list<CAVERNmisc_observer_c *>::iterator i = m_ListOfObservers.begin() ;
       i != m_ListOfObservers.end() ;
       ++i) {
    (*i)->update(this) ;
  }
}

#endif
