/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/

#ifndef CAVERNMISC_HASHDICT_
#define CAVERNMISC_HASHDICT_

#include <stdlib.h>
#include "CAVERNmisc_hash.hxx"
#include "md5Key_c.hxx"

#ifdef CAVERN_THREAD_SAFE
#include "CAVERNts_mutex_c.hxx"
#endif

#ifdef CAVERN_THREAD_SAFE
#define CAVERNMISC_HASHDICT_LOCK	mutex->lock();
#define CAVERNMISC_HASHDICT_UNLOCK mutex->unlock();
#endif

#ifdef NO_THREAD
#define CAVERNMISC_HASHDICT_LOCK
#define CAVERNMISC_HASHDICT_UNLOCK
#endif


template <class valueType, class keyType>
class CAVERNmisc_hashDict;

/*  A dictionary is stored as a collection of entries, each of which
 *  is an CAVERNmisc_hashDictEntry. It contains the key and value for the entry and
 *  a link to create lists of entries. 
 */
template <class valueType, class keyType>
class CAVERNmisc_hashDictEntry {

private:
	keyType		key;
	valueType		value;
	CAVERNmisc_hashDictEntry<valueType,keyType> *	next;

	CAVERNmisc_hashDictEntry(keyType k, valueType v)	{ key = k; value = v; };

	friend class CAVERNmisc_hashDict<valueType,keyType>;
};



/* CAVERNmisc_hashDict: a dictionary mapping (keyType) keys to 
 * (valueType) data pointers.
 * Whatever keyType is, it needs to have a type conversion mechanism to integer.
 * So if you are planning keyType to be some kind of object then that object's class
 * needs an operator int() converter in its class.
 */
template <class valueType, class keyType>
class CAVERNmisc_hashDict {

public:
	CAVERNmisc_hashDict(int entries = 1021);
	~CAVERNmisc_hashDict();


	// Manually lock this array object. (not recommended)
	void lock() {
		CAVERNMISC_HASHDICT_LOCK
	}

	// Manually unlock this array object. (not recommended)
	void unlock() {
		CAVERNMISC_HASHDICT_UNLOCK
	}

	// Calls given routine (passing value) for each entry in dictionary.
	//  The order of entries is not guaranteed to mean anything.
	// If routine returns 0 during an iteration the apply exits.
	// This can be used as an escape hatch.
	void applyToAll(int (*rtn)(keyType key, valueType value) );

	void showHashProfile();

	// This builds a linear array of entries out of the hash dictionary.
	// You must delete the list when done.
	valueType * buildListOfEntries(int *number);

	// Calls the given routine (passing value,data) for each entry in dictionary.
	//  The order of entries is not guaranteed to mean anything.
	// If routine returns 0 during an iteration the applyToAll exits. This can
	// be used as an escape hatch.
	// void *&retdata can be used to pass data into as well as out from the iteration.
	// Note that data passed in must first be type casted to void* to a separate
	// temporary ptr before the ptr is passed to the member function.
	// e.g. void *f;
	// f = (void*) data;
	// applyToAll(xxx,xxx,xxx,xxx, f);
	void applyToAll(int (*rtn)(keyType key, valueType value,
				   void *data1, void *data2, void *data3, void *&retdata),
			void *data1, void *data2, void *data3, void *&retdata );
	// Removes all entries from dictionary.
	void clear();

	/* Enters a key,value pair into the dictionary. Overwrites entry and
	 * returns FALSE if key already exists.
	 */
	int enter(keyType key, valueType value);

	/* Finds entry with given key, setting value to point to value.
	 * Returns FALSE if no such entry.
	 */
	int find(keyType key, valueType &value) const;

	// Removes the entry with the given key. Returns FALSE if no such entry
	int remove(keyType key);

	CAVERNmisc_hashDictEntry<valueType,keyType> *&findEntry(keyType key) const;

	int getNumItems(){return numberOfItems;}
private:

	int numberOfItems;

	// Entries in table
	int tableSize;

	// Entries are stored as an external hash table of CAVERNmisc_hashDictEntry instances.
	CAVERNmisc_hashDictEntry<valueType,keyType> **buckets;

#ifdef CAVERN_THREAD_SAFE
	CAVERNts_mutex_c *mutex;
#endif


};




template <class valueType, class keyType>
CAVERNmisc_hashDict<valueType,keyType>::CAVERNmisc_hashDict( int entries ) {

#ifdef CAVERN_THREAD_SAFE
		mutex = new CAVERNts_mutex_c;
#endif

	tableSize=entries;
	buckets=new CAVERNmisc_hashDictEntry<valueType,keyType> *[tableSize];
	for (int i = 0; i < tableSize; i++) {
		buckets[i] = NULL;
	}
	numberOfItems = 0;
}

template <class valueType, class keyType>
CAVERNmisc_hashDict<valueType,keyType>::~CAVERNmisc_hashDict() {
	clear();
	delete buckets;

#ifdef CAVERN_THREAD_SAFE
		delete mutex;
#endif

}

template <class valueType, class keyType>
void
CAVERNmisc_hashDict<valueType,keyType>::clear() {
	int i;
	CAVERNmisc_hashDictEntry<valueType,keyType> *entry, *nextEntry;
  
	for (i = 0; i < tableSize; i++) {
		for (entry = buckets[i]; entry != NULL; entry = nextEntry) {
			nextEntry = entry->next;
			delete entry;
		}
		buckets[i] = NULL;
	}
	numberOfItems = 0;
}




template <class valueType, class keyType>
int
CAVERNmisc_hashDict<valueType,keyType>::enter(keyType key, valueType value) {
CAVERNmisc_hashDictEntry<valueType,keyType>           *&entry = findEntry(key);
  
	if (entry == NULL) {
		entry = new CAVERNmisc_hashDictEntry<valueType,keyType>(key, value);
		entry->next = NULL;
		numberOfItems++;
		return 1;
	}
	else {
		entry->value = value;
		return 0;
	}
}

template <class valueType, class keyType>
int
CAVERNmisc_hashDict<valueType,keyType>::find(keyType key, valueType &value) const {

	CAVERNmisc_hashDictEntry<valueType,keyType>*&entry = findEntry(key);
  
	if (entry == NULL) {
		value = NULL;
		return 0;
	}
	else {
		value = entry->value;
		return 1;
	}
}

template <class valueType, class keyType>
CAVERNmisc_hashDictEntry<valueType,keyType> *&
CAVERNmisc_hashDict<valueType,keyType>::findEntry(keyType key) const {
	CAVERNmisc_hashDictEntry<valueType,keyType>           **entry;

	/*  
	int *keyFront = (int*) (key.key);

	int copyFront = *keyFront;

	// Make sure keyfront is positive.
	if (copyFront < 0) copyFront = -copyFront;
	*/

	int copyFront = key;

	entry = &buckets[copyFront % tableSize];

	while (*entry != NULL) {
		if ((*entry)->key == key) {
			break;
		}
		entry = &(*entry)->next;
	}
	return *entry;
}

template <class valueType, class keyType>
int
CAVERNmisc_hashDict<valueType,keyType>::remove(keyType key) {
	CAVERNmisc_hashDictEntry<valueType,keyType>           *&entry = findEntry(key);
	CAVERNmisc_hashDictEntry<valueType,keyType>           *tmp;
  
	if (entry == NULL) {
		return 0;
	}
	else {
		tmp = entry;
		entry = entry->next;
		delete tmp;
		numberOfItems--;
		return 1;
	}
}


template <class valueType, class keyType>
valueType*
CAVERNmisc_hashDict<valueType,keyType>::buildListOfEntries(int *number)
{
	int i;
	CAVERNmisc_hashDictEntry<valueType,keyType> *entry;

	valueType *list;

	int numItems = getNumItems();

	if (numItems == 0) {
		*number = 0;
		return NULL;
	}

	list = new valueType[numItems];
	if (list == NULL) {
		*number = 0;
		return NULL;
	}

	*number = numItems;

	int index = 0;

	// Call rtn for each entry in dict
	for (i = 0; i < tableSize; i++) {

		for (entry = buckets[i]; entry != NULL; entry = entry->next) {
			list[index] = entry->value;
			index++;
		}
	}
	return list;
}



// Calls given routine (passing value) for each entry in dictionary.
// The order of entries is not guaranteed to mean anything.
template <class valueType, class keyType>
void
CAVERNmisc_hashDict<valueType,keyType>::applyToAll(int (*rtn)(keyType key, valueType value) ) {
	int i;
	CAVERNmisc_hashDictEntry<valueType,keyType> *entry;

	// Call rtn for each entry in dict
	for (i = 0; i < tableSize; i++) {

		for (entry = buckets[i]; entry != NULL; entry = entry->next) {
			if (((*rtn)(entry->key, entry->value)) == 0) return;
		}
	}
}

template <class valueType, class keyType>
void CAVERNmisc_hashDict<valueType,keyType>::showHashProfile()
{
	int i, count;
	CAVERNmisc_hashDictEntry<valueType,keyType> *entry;

	// Call rtn for each entry in dict
	for (i = 0; i < tableSize; i++) {
		count = 0;
		for (entry = buckets[i]; entry != NULL; entry = entry->next) {
			count++;
		}
		if (count)
		printf("%04d: %d\n",i,count);
	}

}


// Calls given routine (passing value,data) for each entry in dictionary.
// The order of entries is not guaranteed to mean anything.
template <class valueType, class keyType>
void
CAVERNmisc_hashDict<valueType,keyType>::applyToAll(int (*rtn)(keyType key, valueType value,
							    void *data1, void *data2,
							    void *data3, void *&retdata),
						 void *data1, void *data2,
						 void *data3, void *&retdata) {
	int i;
	CAVERNmisc_hashDictEntry<valueType,keyType> *entry;

	// Call rtn for each entry in dict
	for (i = 0; i < tableSize; i++) {

		for (entry = buckets[i]; entry != NULL; entry = entry->next)
			if(((*rtn)(entry->key, entry->value, data1,data2,data3,retdata)) == 0) return;
	}
}


#endif // CAVERNMISC_HASHDICT_ 
