/******************************************************************
 * CAVERNsoft
 * Copyright (C) 1994-2002 Electronic Visualization Laboratory,
 * all rights reserved
 * By Jason Leigh, Yong-joo Cho, Naveen Krishnaprasad, Chris Scharver,
 * Stuart Bailey, Atul Nayak, Shalini Venkataraman, Joshua Eliason,
 * Javid Alimohideen 
 * University of Illinois at Chicago
 * 
 * This publication and its text and code may not be copied for commercial 
 * use without the express written permission of the University of Illinois
 * at Chicago.
 * The contributors disclaim any representation of warranty: use this
 * code at your own risk.
 * Direct questions, comments etc to cavern@evl.uic.edu
 ******************************************************************/
#ifndef _CAVERNDB_SHAREDSTATE_C_HXX
#define _CAVERNDB_SHAREDSTATE_C_HXX

#include <CAVERN.hxx>

/** Encapsulates a shared state to be maintained between multiple users.
    This class provides a base abstract class from which to derives shared state
    information which multiple users encounter.  This state needs to be managed
    by a database client in order to maintain consistency, but the client itself
    is external to the scope of the state.  This state is associated with a
    particular path and key that should not change throughout the life of the
    state.

    Updates are handled through the user of the subject-observer pattern.  The
    shared state observes the client, so when new data arrives, it will unpack
    the state information provided that sharing is enabled.  Sharing is initially
    disabled for initialization purposes, but can be enabled and disabled later
    as required.
    
    Subclasses of the shared state have the responsiblity of handling the format
    of the state for transmission.  The state will need to be packed using the
    packer instance and then sent through the database client.  This process is
    handled by each client because the packing will vary, and the send is
    dependent upon the size of the packed data.  Unpacking state information,
    however, can be done with just the data itself.

    @author cavern@evl.uic.edu
    @version 03/26/2000
*/
class CAVERNdb_sharedState_c : public CAVERNmisc_observer_c
{
	public:
		/// Class constructor.
		CAVERNdb_sharedState_c();

		/** Class constructor.
		    Creating an instance of a shared state requires a reference to a
		    database client which already exists.  The state will be registered
		    with the client as an observer.  Additionally, path and key names
		    are stored within the state for tranmitting and receiving state
		    updates.
		    @param dbclient pointer to an instantiated database client
		    @param path path string to associate with this state
		    @param key key string to associate with this state
		*/
		CAVERNdb_sharedState_c( CAVERNdb_client_c *dbclient, char *path, char *key );
		/// Class destructor.
		virtual ~CAVERNdb_sharedState_c();
		/// Returns the sharing status of this state.
		inline bool isSharingEnabled() { return _sharingEnabled; }
		/// Sets the sharing enabled flag to TRUE.
		inline void enableSharing() { _sharingEnabled = true; }
		/// Sets the sharing enabled flag to FALSE.
		inline void disableSharing() { _sharingEnabled = false; }
		/** Requests a refresh of the data.
		    This method requests the remote state information from the database.
		    It may be useful to call after creating the state.
		*/
		int refresh();
		virtual void update( CAVERNmisc_subject_c *subj );

	protected:
		/** Packs the object state and transmits it to the database server.
		    The state information must be packed into a data buffer and sent
		    to the database server.  The responsibility of packing the state
		    AND sending are placed within this method, so subclasses must
		    put the state information into the database themselves.
		    @return result Result of the attempt to send the state to the server
		*/
		virtual int packAndSendState() = 0;
		/** Unpacks the object state from the data buffer.
		    The state receives a data buffer from the client when an update
		    has occurred, but the state information must be unpacked from
		    that buffer.  This virtual method must be implemented in any
		    derived subclass to extract the appropriate information from
		    the data buffer received from the client.  It is highly advisable
		    to check the data's path, key, and size for security.
		    @param data character data buffer received from the server
		*/
		virtual int unpackState( char *data ) = 0;

	protected:
		/// data packing object
		CAVERNnet_datapack_c _packer;
		/// pointer to a database client which will make the transmissions
		CAVERNdb_client_c *_dbclient;
		/// flag for indicating whether or not sharing is enabled
		bool _sharingEnabled;
		/// path name in the database for storing this state
		char *_path;
		/// key name in the database for storing this state
		char *_key;
};

#endif
